/*
 * This example lets a user's program use the queue interface when running a
 * Rexx script within the program's environment.
 *
 * Note that Regina has external queues. See the manual for the naming
 * conventions. In general, a "@" after the primary name designates the
 * server name and port. Regina has an extended error code for network errors,
 * RXQUEUE_NETERROR, which often means that the server is down or runs on the
 * wrong port.
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterFunctionExe, RexxDeregisterFunction,
 *                 RexxCreateQueue, RexxDeleteQueue, RexxQueryQueue,
 *                 RexxAddQueue, RexxPullQueue
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING, REXXDATETIME
 * You should have read: step2-4.c, step4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN QUEUES IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Function" and "Queue" stuff.
 */
#define INCL_RXFUNC
#define INCL_RXQUEUE
#include "rexx_header.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * Checks whether arg contains ASCending or DESCending and returns
 * 1 for ASCending,
 * 0 for DESCending
 * -1 for non of them.
 */
static int IsAscend( PRXSTRING arg )
{
   static const char *candidate[] = { "ASCENDING", "DESCENDING" };
   int retval, i, max;
   char *p;

   if ( RXNULLSTRING( *arg ) )
      return -1;
   max = (int) RXSTRLEN( *arg );
   p = RXSTRPTR( *arg );
   for ( retval = 0; retval < 2; retval++ )
   {
      if ( ( max >= 3 + retval ) && ( max <= strlen( candidate[retval] ) ) )
      {
         for ( i = 0; i < max; i++ )
            if ( toupper( p[i] ) != candidate[retval][i] )
               break;
            else if ( i + 1 == max )
               return retval;
      }
   }

   return -1;
}

/*
 * This small function is a compare function for the standard function "qsort".
 * It compares two PRXSTRINGs.
 */
static int compRxstring( const void *p1, const void *p2 )
{
   const PRXSTRING r1 = (const PRXSTRING) p1;
   const PRXSTRING r2 = (const PRXSTRING) p2;
   const char *s1, *s2;
   int l1, l2, l, rc;

   s1 = RXSTRPTR( *r1 );
   s2 = RXSTRPTR( *r2 );
   l1 = RXSTRLEN( *r1 );
   l2 = RXSTRLEN( *r2 );
   l = ( l1 < l2 ) ? l1 : l2;

   if ( ( rc = memcmp( s1, s2, l ) ) != 0 )
      return rc;

   if ( l1 < l2 )
      return -1;
   return ( l1 == l2 ) ? 0 : 1;
}

/*
 * Have a look at "RexxFunctionHandler" in the header file.
 * Have a look at step4.c for a full description of the arguments.
 *
 * This "function" accepts one, two or three parameters. The usage is
 * QueueSortDup [ASCending | DESCending] TargetQueue [SourceQueue].
 *
 * This SourceQueue defaults to the current queue's name. The top level buffer
 * of the SourceQueue is copied to the TargetQueue. The TargetQueue's content
 * is sorted either ascending or descending with a default of ascending.
 * The copy happens by reading and write back.
 * The target queue is deleted first.
 */
static APIRET APIENTRY function( PCSZ name,
                                 ULONG argc,
                                 PRXSTRING argv,
                                 PCSZ queuename,
                                 PRXSTRING returnstring )
{
   size_t len;
   int Ascend;
   char *srcq = NULL, *destq = NULL;
   PRXSTRING idx = NULL;
   DATETIME date;
   char *retval = NULL, *p;
   ULONG rc, dup, i, max = 0;

   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return non-zero in this case.
    */
   if ( strcmp( name, "QUEUESORTDUP" ) != 0 )
      return 1;

   if ( argc < 1 )
      return 1;
   if ( ( Ascend = IsAscend( argv ) ) < 0 )
      Ascend = 0;
   else
   {
      argc--;
      argv++;
   }

   if ( ( argc < 1 ) || ( argc > 2 ) )
      return 1;

   len = (size_t) RXSTRLEN( argv[0] );
   if ( RXNULLSTRING( argv[0] ) || ( len == 0 ) )
      return 1;

   if ( ( argc == 2 ) &&
        !RXNULLSTRING( argv[0] ) &&
        ( RXSTRLEN( argv[0] ) == 0 ) )
      return 1;

   printf( "function `%s' entered\n" , name );

   /*
    * Create handy variables for the queues.
    * We need some extra space for the target name, see below at step 2.
    */
   if ( ( destq = malloc( len + 256 ) ) == NULL )
      return 1; /* not really a "SYNTAX" error */
   memcpy( destq, RXSTRPTR( argv[0] ), len );
   destq[len] = '\0';

   if ( ( argc == 2 ) && !RXNULLSTRING( argv[1] ) )
   {
      len = (size_t) RXSTRLEN( argv[1] );
      if ( ( srcq = malloc( len + 1 ) ) == NULL )
      {
         free( destq );
         return 1; /* not really a "SYNTAX" error */
      }
      memcpy( srcq, RXSTRPTR( argv[1] ), len );
      srcq[len] = '\0';

   }
   else
   {
      if ( ( srcq = strdup( queuename ) ) == NULL )
      {
         free( destq );
         return 1; /* not really a "SYNTAX" error */
      }
   }

   /*
    * Now we are prepared for the work.
    * We do this:
    * 1) Delete the target queue and ignore the error if it is RXQUEUE_NOTREG.
    * 2) Create the target queue.
    * 3) Query the source queue, allocate needed space.
    * 4) Pull source elements.
    * 5) Queue back the source and sort the data.
    * 6) Queue the target data.
    * 7) Clean up.
    */


   /*
    * Step 1
    * Arguments of RexxDeleteQueue:
    * PSZ name:   The name of the queue to delete. The value is read only.
    */
   rc = RexxDeleteQueue( destq );
   if ( ( rc != RXQUEUE_OK ) && ( rc != RXQUEUE_NOTREG ) )
   {
      printf( "RexxDeleteQueue(%s) returns %lu\n", destq, rc );
      goto error;
   }

   /*
    * Step 2
    * Arguments of RexxCreateQueue:
    * PSZ createdName:    This buffer will receive the name of the created
    *                     queue on success. It is either the more or less
    *                     similar value of name, or a completely different one.
    *                     "Similar" means a translation to uppercase and
    *                     possibly a hostname resolution for external queues.
    *                     A different name is generated if the desired queue
    *                     name exists already. In this case a queue with an
    *                     unused new name is generated.
    * ULONG createdSize:  Contains the size including the terminating '\0' of
    *                     createdName.
    * PSZ name:           The desired name of the new queue. This may be
    *                     NULL (let the interpreter choose a new name) or
    *                     "@machine[:port]". The later one is a Regina
    *                     extension. This lets the Regina "rxstack" server
    *                     listening on the server named "machine",
    *                     port "port" choose a name.
    * PULONG DupFlag:     Contains either 0 or 1. 0 is returned if the
    *                     desired name was used. 1 is returned if a new
    *                     name was created.
    */
   printf( "(re)creating queue %s...\n", destq );
   rc = RexxCreateQueue( destq,
                         strlen( destq ) + 256,
                         destq,
                         &dup );
   if ( rc != RXQUEUE_OK )
   {
      printf( "RexxCreateQueue(%s) returns %lu\n", destq, rc );
      goto error;
   }
   if ( dup )
   {
      printf("flag=%lu\n",dup);
      printf( "RexxCreateQueue created %s instead of the desired name\n",
              destq );
      goto error;
   }
   printf( "Queue %s created\n", destq );

   /*
    * Step 3
    * Arguments of RexxQueryQueue:
    * PSZ name:      The name of the queue to query.
    * PULONG count:  Filled on success with the number of elements of the
    *                queue.
    */
   rc = RexxQueryQueue( srcq,
                        &max );
   if ( rc != RXQUEUE_OK )
   {
      printf( "RexxQueryQueue(%s) returns %lu\n", srcq, rc );
      goto error;
   }
   printf( "reading %lu elements for sorting\n", max );
   if ( max == 0ul )
   {
      retval = "NO DATA";
      goto error;
   }
   if ( ( idx = malloc( (size_t) max * sizeof(RXSTRING) ) ) == NULL )
   {
      retval = "NO MEMORY";
      goto error;
   }
   memset( idx, 0, (size_t) max * sizeof(RXSTRING) );

   /*
    * Step 4
    * Arguments of RexxPullQueue:
    * PSZ name:        The name of the queue where to pull a line from.
    * PRXSTRING line:  Buffer for the requested line.
    *                  Starting v3.3 Regina checks for a valid RXSTRING and
    *                  uses it if both RXSTRPTR(*line) != NULL and
    *                  RXSTRLEN(*line) indicates sufficient free space.
    *                  Otherwise, and standard up to Regina 3.2, a fresh
    *                  buffer for RXSTRPTR(*line) is allocated and filled; the
    *                  user is responsible to free it later using
    *                  RexxFreeMemory.
    *                  Regina is consistent with Object Rexx in this function
    *                  since version 3.3.
    * PDATETIME date:  Timestamp of the line. This value may be NULL in
    *                  Regina but must be set for other interpreters.
    *                  Note that Regina sets the "valid" flag to 0, which
    *                  means that Regina doesn't maintain a timestamp of
    *                  queue elements for performance reasons.
    * ULONG waitflag:  When reading from an empty buffer, this flag decides
    *                  whether to wait for an arriving line (RXQUEUE_WAIT) or
    *                  return error RXQUEUE_EMPTY at once (RXQUEUE_NOWAIT).
    *                  Regina ignores the waitflag for local queues and
    *                  always returns RXQUEUE_EMPTY at once in this case.
    *                  This is because each thread has its own local queues,
    *                  nothing can feed it up. On the other hand, external
    *                  queues respect this value.
    */
   for ( i = 0; i < max; i++ )
   {
      MAKERXSTRING( idx[i], NULL, 0 );
      rc = RexxPullQueue( srcq,
                          idx + i,
                          &date,
                          RXQUEUE_NOWAIT );
      if ( rc != RXQUEUE_OK )
      {
         printf( "RexxPullQueue(%s) returns %lu on %lu. try\n",
                 srcq,
                 rc,
                 i + 1 );
         goto error;
      }
   }

   /*
    * Step 5
    * We could have queue back the read data at once in the above loop.
    * But it is easier to understand it in the separate loop.
    * Arguments of RexxAddQueue:
    * PSZ name:        The name of the queue where to pull a line from.
    * PRXSTRING line:  The line which has to be placed in the queue.
    *                  The content is not freed by the interpreter.
    *                  Regina accepts strings with RXSTRPTR(*line)==NULL,
    *                  while other interpreters crash. Don't use this feature.
    * ULONG flag:      The flag must be either RXQUEUE_LIFO or RXQUEUE_FIFO
    *                  and replects the position in the queue where the line
    *                  is inserted.
    */
   for ( i = 0; i < max; i++ )
   {
      rc = RexxAddQueue( srcq,
                         idx + i,
                         RXQUEUE_FIFO );
      if ( rc != RXQUEUE_OK )
      {
         printf( "RexxAddQueue(%s) returns %lu on %lu. try\n",
                 srcq,
                 rc,
                 i + 1 );
         goto error;
      }
   }
   qsort( idx, max, sizeof( RXSTRING ), compRxstring );


   /*
    * Step 6
    */
   for ( i = 0; i < max; i++ )
   {
      rc = RexxAddQueue( destq,
                         idx + i,
                         RXQUEUE_FIFO );
      if ( rc != RXQUEUE_OK )
      {
         printf( "RexxAddQueue(%s) returns %lu on %lu. try\n",
                 srcq,
                 rc,
                 i + 1 );
         goto error;
      }
   }
   printf( "sorted data wrote back\n" );

   retval = "OK";
   /*
    * Step 7
    */
error:
   if ( retval == NULL )
   {
      if ( destq != NULL )
         RexxDeleteQueue( destq );
      retval = "ERROR";
   }
   if ( srcq != NULL )
      free( srcq );
   if ( destq != NULL )
      free( destq );
   if ( idx != NULL )
   {
      /*
       * Don't forget to free the space of the lines which was allocated
       * by the interpreter.
       */
      for ( i = 0; i < max; i++ )
         if ( !RXNULLSTRING( idx[i] ) )
            RexxFreeMemory( RXSTRPTR( idx[i] ) );
      free( idx );
   }

   /*
    * The Rexx interpreter has a return buffer of RXAUTOBUFLEN allocated
    * usually. The value is 256 in most cases.
    * 1) We don't rely on this.
    * 2) We are not allowed to free the string if it is too small. It is the
    *    property of the interpreter. It will check for a "lost" string by its
    *    own.
    * 3) Using or replacing the return string is allowed.
    * 4) For good practice we append a not counted ASCII terminator.
    * Regina uses 256 bytes, provides a line terminator but doesn't expect it.
    *
    * Allocating memory should be done with RexxAllocateMemory. This function
    * can be used with Regina and some other interpreters. It is not a known
    * function in IBM's API.
    */
   len = strlen( retval );
   if ( len < RXSTRLEN( *returnstring ) )
      p = RXSTRPTR( *returnstring );
   else
   {
      p = RexxAllocateMemory( len + 1 );
      if ( p == NULL )
         return 1;
   }
   memcpy( p, retval, len );
   p[len] = '\0';
   MAKERXSTRING( *returnstring, p, len );
   return 0;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "queue 1\r\n"
                        "queue 3\r\n"
                        "queue 2\r\n"
                        "queue 'world'\r\n"
                        "queue 'Hello'\r\n"
                        "say 'QueueSortDup(\"Regina\", \"SESSION\") =' QueueSortDup('Regina', 'SESSION')\r\n"
                        "do queued()\r\n"
                        "   parse pull line\r\n"
                        "   queue line\r\n"
                        "   say 'SESSION:' line\r\n"
                        "   end\r\n"
                        "call RxQueue 'Set', 'Regina'\r\n"
                        "do queued()\r\n"
                        "   parse pull line\r\n"
                        "   say 'Regina:' line\r\n"
                        "   end\r\n"
                        "call RxQueue 'Delete', 'Regina'\r\n"
                        "call QueueSortDup 'Regina@', 'SESSION'\r\n"
                        "say 'QueueSortDup(\"Regina@\", \"SESSION\") =' result\r\n"
                        "if result = 'OK' then do\r\n"
                        "   do queued()\r\n"
                        "      parse pull line\r\n"
                        "      queue line\r\n"
                        "      say 'SESSION:' line\r\n"
                        "      end\r\n"
                        "   call RxQueue 'Set', 'Regina@'\r\n"
                        "   do queued()\r\n"
                        "      parse pull line\r\n"
                        "      say 'Regina:' line\r\n"
                        "      end\r\n"
                        "   call RxQueue 'Delete', 'Regina@'\r\n"
                        "   end\r\n";
   RXSTRING instore[2];

   /*
    * The variable pool cannot be used before a script is started.
    * You can use an init exit handler as explained in step3-5.c to get
    * access to the variable pool before a script is executed.
    */

   /*
    * Before you can use a Rexx function hook you have to register the
    * function.
    * We register a function handler in the current executable in opposite to
    * a handler in an external library.
    *
    * ONLY USE RexxRegisterFunctionExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    */
   retval = RexxRegisterFunctionExe( "QueueSortDup",
                                     (PFN) function);
   if ( retval != RXFUNC_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXFUNC_??? values
       * for possible errors.
       */
      printf( "Registering of the function 'DieAfter7Calls' ended with code "
                                                                        "%u\n",
              (unsigned) retval );
      return 1;
   }

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       NULL,         /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   /*
    * We have to deregister the function handler. This isn't really needed
    * before the program's end but provided as an example for proper
    * programming.
    */
   RexxDeregisterFunction( "QueueSortDup" );

   /*
    * The variable pool cannot be used after a script is terminated.
    * You can use a termination exit handler as explained in step3-5.c to get
    * access to the variable pool after a script is executed.
    */

   return (int) ( retval ? retval : rc );
}
