/*
 * This example lets a user's program get access to various information when
 * running a Rexx script within the program's environment.
 *
 * It is likely that the private infomation in the interpreter's variable pool
 * will be different between different interpreter.
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterFunctionExe, RexxDeregisterFunction,
 *                 RexxVariablePool
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING, SHVBLOCK
 * You should have read: step2-4.c, step4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN VARIABLE POOL IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Function" and "Variable" stuff.
 */
#define INCL_RXFUNC
#define INCL_RXSHV
#include "rexx_header.h"
#include <stdio.h>
#include <string.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * assignValue copies the content into value. A fresh buffer is allocated if
 * value's buffer is too small. A terminating '\0' is silently appended.
 * The return value is 1 on success or 0 on error.
 *
 * The Rexx interpreter has a buffer of RXAUTOBUFLEN allocated usually.
 * The value is 256 in most cases.
 * 1) We don't rely on this.
 * 2) We are not allowed to free the string if it is too small. It is the
 *    property of the interpreter. It will check for a "lost" string by its
 *    own.
 * 3) Using or replacing the value's string is allowed.
 * 4) For good practice we append a not counted ASCII terminator.
 * Regina uses 256 bytes, provides a line terminator but doesn't expect it.
 *
 * Allocating memory should be done with RexxAllocateMemory. This function
 * can be used with Regina and some other interpreters. It is not a known
 * function in IBM's API.
 */
static ULONG assignValue( PRXSTRING value, const char *content )
{
   char *p;
   int len = strlen( content );

   if ( RXNULLSTRING( *value ) ||
        ( RXSTRLEN( *value ) <= len ) )
      p = RexxAllocateMemory( len + 1 );
   else
      p = RXSTRPTR( *value );

   if ( p == NULL )
      return 0;

   memcpy( p, content, len );
   MAKERXSTRING( *value, p, len );
   return 1;
}

/*
 * Have a look at "RexxFunctionHandler" in the header file.
 * Have a look at step4.c for a full description of the arguments.
 *
 * This "function" accepts no parameters and returns "nothing" which makes it
 * suitable for CALL instructions only.
 */
static APIRET APIENTRY function( PCSZ name,
                                 ULONG argc,
                                 PRXSTRING argv,
                                 PCSZ queuename,
                                 PRXSTRING returnstring )
{
   SHVBLOCK sb[4];
   ULONG rc;
   int i, max;
   char buf[40], c;

   (argv = argv);
   (queuename = queuename);

   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return non-zero in this case.
    */
   if ( strcmp( name, "SHOWINFO" ) != 0 )
      return 1;

   if ( argc != 0 )
      return 1;

   printf( "various informations:\n" );

   /*
    * The SHVBLOCK structure has several field. Their meaning is:
    * shvnext     The next SHVBLOCK in this request. The last request contains
    *             NULL in this field. There is no limit for the number of
    *             SHVBLOCKs in one request. They may have different request
    *             codes.
    * shvname     The name of the variable to access.
    * shvvalue    The value of the variable.
    * shvnamelen  The length of the buffer in shvname. This value is used for
    *             the codes SHV?? only,
    * shvvaluelen The length of the buffer in shvvalue. This value is used for
    *             the codes SHV?? only,
    * shvcode     Either RXSHV_SET, RXSHV_FETCH, RXSHV_DROPV, RXSHV_SYSET,
    *             RXSHV_SYFET, RXSHV_SYDRO, RXSHV_NEXTV, RXSHV_PRIV, RXSHV_EXIT
    * shvret      The return code of this request,
    *             RXSHV_OK:    Everything's OK
    *             RXSHV_NEWV:  Variable unknown, default returned
    *             RXSHV_LVAR:  Last variable in RXSHV_NEXTV request.
    *             RXSHV_TRUNC: Buffer too small, name or value truncated.
    *             RXSHV_BADN:  Illegal characters in name.
    *             RXSHV_MEMFL: Not enough free memory.
    *             RXSHV_BADF:  Invalid function code.
    */

   /*
    * Reading "private" variables needs these settings:
    * shvname:     Must be set to the variable's name.
    * shvvalue:    Either set shvvalue.strptr to NULL in which case a fresh
    *              string is allocated by the interpreter which should be freed
    *              by RexxFreeMemory or set shvvalue.strptr to a buffer and
    *              put the buffer's length into shvvaluelen.
    *              The value of the variable is put into this buffer and
    *              shvvalue.strlength is set to the correct length. The string
    *              is terminated in Regina but not in all other Rexx
    *              interpreters.
    *              Before calling RexxVariablePool you should set
    *              shvvalue.strlength to shvvaluelen if you are not using
    *              Regina.
    * shvnamelen:  ignored
    * shvvaluelen: Official SAA documentation:
    *              ignored if shvvalue.strptr == NULL, otherwise this value
    *              is read by the interpreter to determine the maximum size
    *              of the buffer in shvvalue.strptr.
    *              ATTENTION:
    *              Due to bugs in the OS/2's REXX interpreter most others
    *              aim to do the same as this buggy interpreter. Therefore
    *              this happens (at least with Regina and Object Rexx):
    *              On input this value is read to determine the maximum
    *              buffer size of shvvalue.strptr.
    *              On output the value is set to the length of the data
    *              copied to shvvalue.strptr which equals shvvalue.strlength.
    * shvcode:     RXSHV_PRIV
    * shvret:      Either RXSHV_OK or a combination of RXSHV_NEWV, RXSHV_BADN,
    *              (RXSHV_MEMFL or RXSHV_TRUNC).
    *              MEMFL is returned only if shvvalue.strptr was NULL and
    *              the allocation doesn't complete successfully, TRUNC is
    *              return is the string's length is smaller than shvvaluelen.
    *
    * Possible variable names in Regina:
    * 1) PARM
    *    returns a non-negative number. If not 0, the number is the
    *    maximum addressable number of PARM.<number>
    * 2) PARM.<number>
    *    returns the number'th argument of the current script. Number must be
    *    a non-negative number. Missing arguments or arguments beyond the last
    *    possible argument are returned as NULL pointers.
    *    NOTE: The arguments are not taken from the current function, they are
    *          taken from the script's invocation.
    * 3) QUENAME
    *    returns the current default queue. This value should be the same as
    *    the "queuename" parameter of a function invocation.
    * 4) VERSION
    *    returns the same as that from "PARSE VERSION".
    * 5) SOURCE
    *    returns the same as that from "PARSE SOURCE".
    */

   MAKERXSTRING( sb[0].shvvalue, NULL, 0 );
   sb[0].shvnamelen = 0;
   sb[0].shvvaluelen = 0;
   sb[0].shvcode = RXSHV_PRIV;

   sb[0].shvnext = sb + 1;
   MAKERXSTRING( sb[0].shvname, "SOURCE", 6 );

   sb[1] = sb[0];
   sb[1].shvnext = sb + 2;
   MAKERXSTRING( sb[1].shvname, "QUENAME", 7 );

   sb[2] = sb[0];
   sb[2].shvnext = sb + 3;
   MAKERXSTRING( sb[2].shvname, "VERSION", 7 );

   sb[3] = sb[0];
   sb[3].shvnext = NULL;
   MAKERXSTRING( sb[3].shvname, "PARM", 4 );

   rc = RexxVariablePool( sb );
   if ( rc == RXSHV_NOAVL )
   {
      printf( "The variable pool isn't available\n ");
      return 1; /* Not really a SYNTAX error */
   }

   for ( i = 0; i < 4; i++ )
   {
      sprintf( buf, "%s:", RXSTRPTR( sb[i].shvname ) );
      if ( sb[i].shvret != RXSHV_OK )
      {
         printf( "%-10s ERROR code 0x%02X\n",
                 buf,
                 sb[i].shvret );
         strcpy( buf, "" );
      }
      else
      {
         printf( "%-10s %.*s\n",
                 buf,
                 (int) RXSTRLEN( sb[i].shvvalue ), RXSTRPTR( sb[i].shvvalue ) );
         /*
          * This trick copies the content of the last value (that from
          * PARM) to buf. We can use it below.
          */
         if ( ( max = (int) RXSTRLEN( sb[i].shvvalue ) ) > 39 )
            max = 0;
         sprintf( buf, "%.*s", max, RXSTRPTR( sb[i].shvvalue ) );
      }
      if ( !RXNULLSTRING( sb[i].shvvalue ) )
         RexxFreeMemory( RXSTRPTR( sb[i].shvvalue ) );
   }

   /*
    * buf contains the number of possible parameters.
    */
   if ( sscanf( buf, "%d %c", &max, &c ) == 1 )
   {
      for ( i = 1; i <= max; i++ )
      {
         sb[0].shvnext = NULL;
         sprintf( buf, "PARM.%d", i );
         MAKERXSTRING( sb[0].shvname, buf, strlen( buf ) );
         MAKERXSTRING( sb[0].shvvalue, NULL, 0 );
         sb[0].shvnamelen = 0;
         sb[0].shvvaluelen = 0;
         sb[0].shvcode = RXSHV_PRIV;
         RexxVariablePool( sb ); /* RXSHV_NOAVL cannot happen */

         sprintf( buf, "PARM.%d:", i );
         if ( sb[0].shvret != RXSHV_OK )
            printf( "%-10s ERROR code 0x%02X\n",
                    buf,
                    sb[0].shvret );
         else
            printf( "%-10s %.*s\n",
                    buf,
                    (int) RXSTRLEN( sb[0].shvvalue ), RXSTRPTR( sb[0].shvvalue ) );
         if ( !RXNULLSTRING( sb[0].shvvalue ) )
            RexxFreeMemory( RXSTRPTR( sb[0].shvvalue ) );
      }
   }

   assignValue( returnstring, "OK" );
   return 0;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "parse version v\r\n"
                        "say 'PARSE VERSION returns' v\r\n"
                        "parse source s\r\n"
                        "say 'PARSE SOURCE returns' s\r\n"
                        "say 'calling ShowInfo'\r\n"
                        "call ShowInfo\r\n"
                         "say 'calling proc 1, 2'\r\n"
                        "call proc 1, 2\r\n"
                        "return\r\n"
                        "proc: procedure\r\n"
                        " say 'calling ShowInfo'\r\n"
                        " call ShowInfo\r\n"
                        " return\r\n";
   RXSTRING instore[2];
   RXSTRING args[2];

   /*
    * The variable pool cannot be used before a script is started.
    * You can use an init exit handler as explained in step3-5.c to get
    * access to the variable pool before a script is executed.
    */

   /*
    * Before you can use a Rexx function hook you have to register the
    * function.
    * We register a function handler in the current executable in opposite to
    * a handler in an external library.
    *
    * ONLY USE RexxRegisterFunctionExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    */
   retval = RexxRegisterFunctionExe( "ShowInfo",
                                     (PFN) function);
   if ( retval != RXFUNC_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXFUNC_??? values
       * for possible errors.
       */
      printf( "Registering of the function 'DieAfter7Calls' ended with code "
                                                                        "%u\n",
              (unsigned) retval );
      return 1;
   }

   MAKERXSTRING( args[0], "argument 1 to the script", 24 );
   MAKERXSTRING( args[1], "argument 2 to the script", 24 );
   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 2,            /* ArgCount */
                       args,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXSUBROUTINE, /* CallType, must not be RXCOMMAND if
                                      * using more than one argument
                                      */
                       NULL,         /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   /*
    * We have to deregister the function handler. This isn't really needed
    * before the program's end but provided as an example for proper
    * programming.
    */
   RexxDeregisterFunction( "ShowInfo" );

   /*
    * The variable pool cannot be used after a script is terminated.
    * You can use a termination exit handler as explained in step3-5.c to get
    * access to the variable pool after a script is executed.
    */

   return (int) ( retval ? retval : rc );
}
