/*
 * This example lets a user's program use the variable access interface to
 * iterate every accessible variable of one procedure level within the
 * program's environment.
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterFunctionExe, RexxDeregisterFunction,
 *                 RexxVariablePool
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING, SHVBLOCK
 * You should have read: step2-4.c, step4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN VARIABLE POOL IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Function" and "Variable" stuff.
 */
#define INCL_RXFUNC
#define INCL_RXSHV
#include "rexx_header.h"
#include <stdio.h>
#include <string.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * assignValue copies the content into value. A fresh buffer is allocated if
 * value's buffer is too small. A terminating '\0' is silently appended.
 * The return value is 1 on success or 0 on error.
 *
 * The Rexx interpreter has a buffer of RXAUTOBUFLEN allocated usually.
 * The value is 256 in most cases.
 * 1) We don't rely on this.
 * 2) We are not allowed to free the string if it is too small. It is the
 *    property of the interpreter. It will check for a "lost" string by its
 *    own.
 * 3) Using or replacing the value's string is allowed.
 * 4) For good practice we append a not counted ASCII terminator.
 * Regina uses 256 bytes, provides a line terminator but doesn't expect it.
 *
 * Allocating memory should be done with RexxAllocateMemory. This function
 * can be used with Regina and some other interpreters. It is not a known
 * function in IBM's API.
 */
static ULONG assignValue( PRXSTRING value, const char *content )
{
   char *p;
   int len = strlen( content );

   if ( RXNULLSTRING( *value ) ||
        ( RXSTRLEN( *value ) <= len ) )
      p = RexxAllocateMemory( len + 1 );
   else
      p = RXSTRPTR( *value );

   if ( p == NULL )
      return 0;

   memcpy( p, content, len );
   MAKERXSTRING( *value, p, len );
   return 1;
}

/*
 * Have a look at "RexxFunctionHandler" in the header file.
 * Have a look at step4.c for a full description of the arguments.
 *
 * This "function" accepts no parameters and returns "nothing" which makes it
 * suitable for CALL instructions only.
 */
static APIRET APIENTRY function( PCSZ name,
                                 ULONG argc,
                                 PRXSTRING argv,
                                 PCSZ queuename,
                                 PRXSTRING returnstring )
{
   SHVBLOCK sb;
   ULONG rc;

   (argv = argv);
   (queuename = queuename);

   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return non-zero in this case.
    */
   if ( strcmp( name, "SHOWVARS" ) != 0 )
      return 1;

   if ( argc != 0 )
      return 1;

   printf( "known variables in the function ShowVars:\n" );

   /*
    * Iterating every known value is done using a structure.
    *
    * The SHVBLOCK structure has several field. Their meaning is:
    * shvnext     The next SHVBLOCK in this request. The last request contains
    *             NULL in this field. There is no limit for the number of
    *             SHVBLOCKs in one request. They may have different request
    *             codes.
    * shvname     The name of the variable to access.
    * shvvalue    The value of the variable.
    * shvnamelen  The length of the buffer in shvname. This value is used for
    *             the codes SHV?? only,
    * shvvaluelen The length of the buffer in shvvalue. This value is used for
    *             the codes SHV?? only,
    * shvcode     Either RXSHV_SET, RXSHV_FETCH, RXSHV_DROPV, RXSHV_SYSET,
    *             RXSHV_SYFET, RXSHV_SYDRO, RXSHV_NEXTV, RXSHV_PRIV, RXSHV_EXIT
    * shvret      The return code of this request,
    *             RXSHV_OK:    Everything's OK
    *             RXSHV_NEWV:  Variable unknown, default returned
    *             RXSHV_LVAR:  Last variable in RXSHV_NEXTV request.
    *             RXSHV_TRUNC: Buffer too small, name or value truncated.
    *             RXSHV_BADN:  Illegal characters in name.
    *             RXSHV_MEMFL: Not enough free memory.
    *             RXSHV_BADF:  Invalid function code.
    */

   /*
    * Iterating variables needs these settings:
    * shvname:     Either set shvname.strptr to NULL in which case a fresh
    *              string is allocated by the interpreter which should be freed
    *              by RexxFreeMemory or set shvname.strptr to a buffer and
    *              put the buffer's length into shvnamelen.
    *              The name of the variable is put into this buffer and
    *              shvname.strlength is set to the correct length. The string
    *              is terminated in Regina but not in all other Rexx
    *              interpreters.
    *              Before calling RexxVariablePool you should set
    *              shvname.strlength to shvnamelen if you are not using
    *              Regina.
    * shvvalue:    Either set shvvalue.strptr to NULL in which case a fresh
    *              string is allocated by the interpreter which should be freed
    *              by RexxFreeMemory or set shvvalue.strptr to a buffer and
    *              put the buffer's length into shvvaluelen.
    *              The value of the variable is put into this buffer and
    *              shvvalue.strlength is set to the correct length. The string
    *              is terminated in Regina but not in all other Rexx
    *              interpreters.
    *              Before calling RexxVariablePool you should set
    *              shvvalue.strlength to shvvaluelen if you are not using
    *              Regina.
    * shvnamelen:  Official SAA documentation:
    *              ignored if shvname.strptr == NULL, otherwise this value
    *              is read by the interpreter to determine the maximum size
    *              of the buffer in shvname.strptr.
    *              ATTENTION:
    *              Due to bugs in the OS/2's REXX interpreter most others
    *              aim to do the same as this buggy interpreter. Therefore
    *              this happens (at least with Regina and Object Rexx):
    *              On input this value is read to determine the maximum
    *              buffer size of shvname.strptr.
    *              On output the value is set to the length of the data
    *              copied to shvname.strptr which equals shvname.strlength.
    * shvvaluelen: Official SAA documentation:
    *              ignored if shvvalue.strptr == NULL, otherwise this value
    *              is read by the interpreter to determine the maximum size
    *              of the buffer in shvvalue.strptr.
    *              ATTENTION:
    *              Due to bugs in the OS/2's REXX interpreter most others
    *              aim to do the same as this buggy interpreter. Therefore
    *              this happens (at least with Regina and Object Rexx):
    *              On input this value is read to determine the maximum
    *              buffer size of shvvalue.strptr.
    *              On output the value is set to the length of the data
    *              copied to shvvalue.strptr which equals shvvalue.strlength.
    * shvcode:     RXSHV_NEXTV
    * shvret:      Either RXSHV_OK or a combination of RXSHV_MEMFL and
    *              RXSHV_TRUNC. The last variable's shvret value is set to
    *              RXSHV_LVAR and the variable's name and value is set to NULL.
    *              MEMFL is returned only if one of shvvalue.strptr or
    *              shvname.strptr was NULL and the allocation doesn't complete
    *              successfully, TRUNC is return is the string's length is
    *              smaller than shvvaluelen or shvnamelen.
    *
    * WARNINGS:
    * 1) You should iterate until RXSHV_LVAR is returned. The interpreter
    *    maintains an internal iterator which knows or thinks it knows the
    *    state of the iteration. An iteration is stopped either when RXSHV_LVAR
    *    is returned or when using RexxVariablePool for set/read/drop access or
    *    after the end of the current handler. The current handler may be
    *    one of the exit handlers, the function handlers, the command handlers.
    * 2) It is bad programming style to use this technique at all. The
    *    iteration may cost some time in big programs with hundreds of
    *    variables without a PROCEDURE instruction. Use RXSHV_NEXTV for
    *    debugging only.
    * 3) Don't expect an ordering of the variables.
    * 4) Regina includes stem's default values if they are assigned. Other
    *    interpreters may omit this.
    */

   /*
    * We iterate through the pool and let each value be allocated freshly by
    * the interpreter. If maximum boundaries of the string lengths can be
    * used, a provided buffer is much faster for iteration.
    */

   sb.shvnext = NULL;
   sb.shvnamelen = 0;
   sb.shvvaluelen = 0;
   sb.shvcode = RXSHV_NEXTV;


   for ( ; ; )
   {
      MAKERXSTRING( sb.shvname, NULL, 0 );
      MAKERXSTRING( sb.shvvalue, NULL, 0 );
      /*
       * The return of RexxVariablePool is either RXSHV_NOAVL or the or'ed
       * combination of all shvret values.
       */
      rc = RexxVariablePool( &sb );
      if ( rc == RXSHV_NOAVL )
      {
         printf( "The variable pool isn't available\n ");
         return 1; /* Not really a SYNTAX error */
      }

      if ( rc == RXSHV_LVAR )
         break;

      if ( rc != RXSHV_OK )
      {
         printf( "Iterating returns 0x%02lX\n", rc );
         assignValue( returnstring, "ERROR" );
         return 0;
      }
      printf( "  value of `%.*s' is `%.*s'\n",
              (int) RXSTRLEN( sb.shvname ), RXSTRPTR( sb.shvname ),
              (int) RXSTRLEN( sb.shvvalue ), RXSTRPTR( sb.shvvalue ) );

      if ( !RXNULLSTRING( sb.shvname ) )
         RexxFreeMemory( RXSTRPTR( sb.shvname ) );
      if ( !RXNULLSTRING( sb.shvvalue ) )
         RexxFreeMemory( RXSTRPTR( sb.shvvalue ) );
   }

   assignValue( returnstring, "OK" );
   return 0;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "say 'setting A=1, B=2, C.=3, C.A=4'\r\n"
                        "A=1; B=2; C.=3; C.A=4\r\n"
                        "say 'calling ShowVars'\r\n"
                        "call ShowVars\r\n"
                        "call proc\r\n"
                        "say 'back in the main procedure calling ShowVars'\r\n"
                        "call ShowVars\r\n"
                        "return\r\n"
                        "proc: procedure expose A C.A\r\n"
                        " say 'in a procedure exposing A and C.A modifying B'\r\n"
                        " B='Hello'\r\n"
                        " say 'calling ShowVars'\r\n"
                        " call ShowVars\r\n"
                        " return\r\n";
   RXSTRING instore[2];

   /*
    * The variable pool cannot be used before a script is started.
    * You can use an init exit handler as explained in step3-5.c to get
    * access to the variable pool before a script is executed.
    */

   /*
    * Before you can use a Rexx function hook you have to register the
    * function.
    * We register a function handler in the current executable in opposite to
    * a handler in an external library.
    *
    * ONLY USE RexxRegisterFunctionExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    */
   retval = RexxRegisterFunctionExe( "ShowVars",
                                     (PFN) function);
   if ( retval != RXFUNC_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXFUNC_??? values
       * for possible errors.
       */
      printf( "Registering of the function 'DieAfter7Calls' ended with code "
                                                                        "%u\n",
              (unsigned) retval );
      return 1;
   }

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       NULL,         /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   /*
    * We have to deregister the function handler. This isn't really needed
    * before the program's end but provided as an example for proper
    * programming.
    */
   RexxDeregisterFunction( "ShowVars" );

   /*
    * The variable pool cannot be used after a script is terminated.
    * You can use a termination exit handler as explained in step3-5.c to get
    * access to the variable pool after a script is executed.
    */

   return (int) ( retval ? retval : rc );
}
