/*
 * This example lets a user's program use the variable access interface for
 * direct use within the program's environment. The direct access bypasses
 * any further interpretation of variable names and is much faster for
 * stem names if it can be guaranteed that the used names need no tail
 * expansion.
 * See step8-1.c for the normal (symbolic) use.
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterFunctionExe, RexxDeregisterFunction,
 *                 RexxVariablePool
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING, SHVBLOCK
 * You should have read: step2-4.c, step4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN VARIABLE POOL IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Function" and "Variable" stuff.
 */
#define INCL_RXFUNC
#define INCL_RXSHV
#include "rexx_header.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * assignValue copies the content into value. A fresh buffer is allocated if
 * value's buffer is too small. A terminating '\0' is silently appended.
 * The return value is 1 on success or 0 on error.
 *
 * The Rexx interpreter has a buffer of RXAUTOBUFLEN allocated usually.
 * The value is 256 in most cases.
 * 1) We don't rely on this.
 * 2) We are not allowed to free the string if it is too small. It is the
 *    property of the interpreter. It will check for a "lost" string by its
 *    own.
 * 3) Using or replacing the value's string is allowed.
 * 4) For good practice we append a not counted ASCII terminator.
 * Regina uses 256 bytes, provides a line terminator but doesn't expect it.
 *
 * Allocating memory should be done with RexxAllocateMemory. This function
 * can be used with Regina and some other interpreters. It is not a known
 * function in IBM's API.
 */
static ULONG assignValue( PRXSTRING value, const char *content )
{
   char *p;
   int len = strlen( content );

   if ( RXNULLSTRING( *value ) ||
        ( RXSTRLEN( *value ) <= len ) )
      p = RexxAllocateMemory( len + 1 );
   else
      p = RXSTRPTR( *value );

   if ( p == NULL )
      return 0;

   memcpy( p, content, len );
   MAKERXSTRING( *value, p, len );
   return 1;
}

/*
 * Have a look at "RexxFunctionHandler" in the header file.
 * Have a look at step4.c for a full description of the arguments.
 *
 * This "function" accepts one or two parameters. The first is the stem's name,
 * the second the new value. Each value is dropped if the second argument is
 * missing.
 * This function reads the stem's ".0" value ( end) and assigns the new value
 * to "stem".1 .. "stem".end
 */
static APIRET APIENTRY function( PCSZ name,
                                 ULONG argc,
                                 PRXSTRING argv,
                                 PCSZ queuename,
                                 PRXSTRING returnstring )
{
   SHVBLOCK sb;
   size_t len;
   int i, max;
   char *varName, buf[40], c;

   (queuename = queuename);

   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return non-zero in this case.
    */
   if ( strcmp( name, "SETSTEM" ) != 0 )
      return 1;

   if ( ( argc < 1 ) || ( argc > 2 ) )
      return 1;
   len = (size_t) RXSTRLEN( argv[0] );
   if ( RXNULLSTRING( argv[0] ) || ( len == 0 ) )
      return 1;

   printf( "function `%s' entered\n" , name );

   /*
    * The used variable name is our own property. We don't have to allocate
    * it with RexxAllocateMemory, even if we use it further below.
    * The extra buffer should be enough for ".VeryLongNumber".
    */
   if ( ( varName = malloc( len + 40 ) ) == NULL )
      return 1; /* not really a "SYNTAX" error */

   memcpy( varName, RXSTRPTR( argv[0] ), len );
   /*
    * Add a missing trailing dot. Then add a zero.
    */
   if ( varName[len - 1] != '.' )
      varName[len++] = '.';
   varName[len] = '0';
   varName[len + 1] = '\0';

   /*
    * Now read the value directly. This means that the variable name is
    * NOT converted to uppercase and intermediate values ar NOTe
    * resolved.
    * Unlike the symbolic version, reading "x.b" will NOT result in "7",
    * if "x.2" was set to 7 and "b" to "2". To get "7" you must pass in
    * "X.2".
    */

   /*
    * The SHVBLOCK structure has several field. Their meaning is:
    * shvnext     The next SHVBLOCK in this request. The last request contains
    *             NULL in this field. There is no limit for the number of
    *             SHVBLOCKs in one request. They may have different request
    *             codes.
    * shvname     The name of the variable to access.
    * shvvalue    The value of the variable.
    * shvnamelen  The length of the buffer in shvname. This value is used for
    *             the codes SHV?? only,
    * shvvaluelen The length of the buffer in shvvalue. This value is used for
    *             the codes SHV?? only,
    * shvcode     Either RXSHV_SET, RXSHV_FETCH, RXSHV_DROPV, RXSHV_SYSET,
    *             RXSHV_SYFET, RXSHV_SYDRO, RXSHV_NEXTV, RXSHV_PRIV, RXSHV_EXIT
    * shvret      The return code of this request,
    *             RXSHV_OK:    Everything's OK
    *             RXSHV_NEWV:  Variable unknown, default returned
    *             RXSHV_LVAR:  Last variable in RXSHV_NEXTV request.
    *             RXSHV_TRUNC: Buffer too small, name or value truncated.
    *             RXSHV_BADN:  Illegal characters in name.
    *             RXSHV_MEMFL: Not enough free memory.
    *             RXSHV_BADF:  Invalid function code.
    */

   /*
    * Reading a variable needs these settings:
    * shvname:     Must be set the the variable's name.
    * shvvalue:    Either set shvvalue.strptr to NULL in which case a fresh
    *              string is allocated by the interpreter which should be freed
    *              by RexxFreeMemory or set shvvalue.strptr to a buffer and
    *              put the buffer's length into shvvaluelen.
    *              The value of the variable is put into this buffer and
    *              shvvalue.strlength is set to the correct length. The string
    *              is terminated in Regina but not in all other Rexx
    *              interpreters.
    *              Before calling RexxVariablePool you should set
    *              shvvalue.strlength to shvvaluelen if you are not using
    *              Regina.
    * shvnamelen:  ignored
    * shvvaluelen: Official SAA documentation:
    *              ignored if shvvalue.strptr == NULL, otherwise this value
    *              is read by the interpreter to determine the maximum size
    *              of the buffer in shvvalue.strptr.
    *              ATTENTION:
    *              Due to bugs in the OS/2's REXX interpreter most others
    *              aim to do the same as this buggy interpreter. Therefore
    *              this happens (at least with Regina and Object Rexx):
    *              On input this value is read to determine the maximum
    *              buffer size of shvvalue.strptr.
    *              On output the value is set to the length of the data
    *              copied to shvvalue.strptr which equals shvvalue.strlength.
    * shvcode:     RXSHV_FETCH
    * shvret:      Either RXSHV_OK or a combination of RXSHV_NEWV, RXSHV_BADN,
    *              (RXSHV_MEMFL or RXSHV_TRUNC).
    *              MEMFL is returned only if shvvalue.strptr was NULL and
    *              the allocation doesn't complete successfully, TRUNC is
    *              return is the string's length is smaller than shvvaluelen.
    */

   sb.shvnext = NULL;
   MAKERXSTRING( sb.shvname, varName, strlen( varName ) );
   MAKERXSTRING( sb.shvvalue, buf, sizeof( buf ) - 1 );
   sb.shvvaluelen = sizeof( buf ) - 1; /* Providing a buffer is MUCH faster */
   sb.shvcode = RXSHV_FETCH;

   /*
    * The return of RexxVariablePool is either RXSHV_NOAVL or the or'ed
    * combination of all shvret values.
    */
   if ( RexxVariablePool( &sb ) == RXSHV_NOAVL )
   {
      printf( "The variable pool isn't available\n ");
      return 1; /* Not really a SYNTAX error */
   }

   /*
    * We accept the default value of the stem:
    */
   sb.shvret = (UCHAR) ( sb.shvret & ~RXSHV_NEWV );
   if ( sb.shvret != RXSHV_OK )
   {
      printf( "Accessing %s returns 0x%02X\n", varName, sb.shvret );
      assignValue( returnstring, "ERROR" );
      return 0;
   }

   /*
    * Interpret the value as a number.
    *
    * Note that OS/2 REXX errorneously changes shvvaluelen instead of
    * shvvalue.strlength turning their official doc upside down.
    */
   if ( RXSTRLEN( sb.shvvalue ) < sb.shvvaluelen )
      buf[RXSTRLEN( sb.shvvalue )] = '\0'; /* We'd let one byte free */
   else
      buf[sb.shvvaluelen] = '\0'; /* We'd let one byte free */

   if ( sscanf( buf, "%d %c", &max, &c ) != 1 )
   {
      printf( "The content of %s, `%s' must be a number\n", varName, buf );
      assignValue( returnstring, "ERROR" );
      return 0;
   }
   if ( max < 0 )
   {
      printf( "The content of %s, `%s' must be a non-negative number\n",
              varName, buf );
      assignValue( returnstring, "ERROR" );
      return 0;
   }

   /*
    * Now access each stem's leaf and set it to the given value. Either drop
    * it if the second argument is not given or set it to that argument.
    */

   /*
    * Dropping a variable needs these settings:
    * shvname:     Must be set the the variable's name.
    * shvvalue:    ignored
    * shvnamelen:  ignored
    * shvvaluelen: ignored
    * shvcode:     RXSHV_DROPV
    * shvret:      Either RXSHV_OK or RXSHV_BADN possibly or'ed with RXSHV_NEWV.
    */

   /*
    * Setting a variable needs these settings:
    * shvname:     Must be set the the variable's name.
    * shvvalue:    Must be set the the variable's value.
    * shvnamelen:  ignored
    * shvvaluelen: must be equal to shvvalue.strlength for other interpreters,
    *              ignored by Regina.
    * shvcode:     RXSHV_SET
    * shvret:      Either RXSHV_OK or RXSHV_BADN possibly or'ed with RXSHV_NEWV.
    */
   if ( ( argc < 2 ) || RXNULLSTRING( argv[1] ) )
   {
      sb.shvnext = NULL;
      /*
       * shvname is set later.
       */
      sb.shvcode = RXSHV_DROPV;
   }
   else
   {
      sb.shvnext = NULL;
      /*
       * shvname is set later.
       */
      sb.shvvalue = argv[1];
      sb.shvvaluelen = RXSTRLEN( sb.shvvalue ) - 1;
      sb.shvcode = RXSHV_SET;
   }

   for ( i = 1; i <= max; i++ )
   {
      sprintf( varName + len, "%d", i );
      MAKERXSTRING( sb.shvname, varName, strlen( varName ) );
      /*
       * The return code of RexxVariablePool can't be RXSHV_NOAVL.
       */
      RexxVariablePool( &sb );
      /*
       * We accept the usage of currently unassigned variables.
       */
      sb.shvret = (UCHAR) ( sb.shvret & ~RXSHV_NEWV );
      if ( sb.shvret != RXSHV_OK )
      {
         printf( "%s %s returns 0x%02X\n",
                 ( sb.shvcode == RXSHV_SET ) ? "Writing" : "Dropping",
                 varName,
                 sb.shvret );
         assignValue( returnstring, "ERROR" );
         return 0;
      }
   }

   assignValue( returnstring, "OK" );
   return 0;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "x.7.0 = 3\r\n"
                        "x.7.2 = 'Hello'\r\n"
                        "say 'ATTENTION: Direct variable access inhibits the'\r\n"
                        "say 'interpreter from uppercasing characters, too.'\r\n"
                        "say ''\r\n"
                        "say 'setting A to 7'\r\n"
                        "a = 7\r\n"
                        "say 'SetStem(\"x.a\") returns' setStem('x.a')\r\n"
                        "say 'SetStem(\"X.A\") returns' setStem('X.A')\r\n"
                        "say 'SetStem(\"X.7\") returns' setStem('X.7')\r\n"
                        "do i = 1 to x.a.0\r\n"
                        "   say 'X.A.'i '=' x.a.i\r\n"
                        "   end\r\n"
                        "x.a.2 = 'Hello'\r\n"
                        "say 'SetStem(\"x.a\", \"y\") returns' setStem('x.a', 'y')\r\n"
                        "say 'SetStem(\"X.A\", \"y\") returns' setStem('X.A', 'y')\r\n"
                        "say 'SetStem(\"X.7\", \"y\") returns' setStem('X.7', 'y')\r\n"
                        "do i = 1 to x.a.0\r\n"
                        "   say 'X.A.'i '=' x.a.i\r\n"
                        "   end\r\n";
   RXSTRING instore[2];

   /*
    * The variable pool cannot be used before a script is started.
    * You can use an init exit handler as explained in step3-5.c to get
    * access to the variable pool before a script is executed.
    */

   /*
    * Before you can use a Rexx function hook you have to register the
    * function.
    * We register a function handler in the current executable in opposite to
    * a handler in an external library.
    *
    * ONLY USE RexxRegisterFunctionExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    */
   retval = RexxRegisterFunctionExe( "SetStem",
                                     (PFN) function);
   if ( retval != RXFUNC_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXFUNC_??? values
       * for possible errors.
       */
      printf( "Registering of the function 'DieAfter7Calls' ended with code "
                                                                        "%u\n",
              (unsigned) retval );
      return 1;
   }

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       NULL,         /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   /*
    * We have to deregister the function handler. This isn't really needed
    * before the program's end but provided as an example for proper
    * programming.
    */
   RexxDeregisterFunction( "SetStem" );

   /*
    * The variable pool cannot be used after a script is terminated.
    * You can use a termination exit handler as explained in step3-5.c to get
    * access to the variable pool after a script is executed.
    */

   return (int) ( retval ? retval : rc );
}
