/*
 * This example lets a user's program do a callback from an invoked function
 * of a Rexx script back into the original script within the program's
 * environment.
 * This short pseudo-code may explain it:
 *
 * A) Execute script
 * B) call registered_function
 * C) call helper_builtin_function_in_script_as_callback
 *
 * The advantages:
 * a) Registered_function gets access to the variables of the script.
 * b) The user can execute some code in Rexx with the same variable set as
 *    the original script.
 *
 * Point a) can be managed by RexxVariablePool, step8-?, but point b) can't
 * be emulated by other techniques.
 * A typical usage is an event driven system like GUI systems, which reside
 * in an external function package, but wants the script writing user to take
 * advantage of some events like mouse clicks.
 *
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterFunctionExe, RexxDeregisterFunction,
 *                 RexxCallBack
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING
 * You should have read: step2-4.c, step4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE. RexxCallBack IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN FUNCTION HANDLER SET IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Function" and "Asynchroneous Request
 * Interface" stuff.
 */
#define INCL_RXFUNC
#define INCL_RXARI
#include "rexx_header.h"
#include <stdio.h>
#include <string.h>
#include <time.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * DoCallback tries to pass arg1 as the single argument to the function with
 * the given name.
 */
void DoCallback( const char *function, const char *arg1 )
{
   char buf[RXAUTOBUFLEN];
   RXSTRING ArgList[1];
   SHORT ReturnCode;
   RXSTRING Result;
   ULONG rc;

   /*
    * First, build all arguments for the callback. Arguments are not changed
    * by the interpreter.
    */
   MAKERXSTRING( ArgList[0], (char *) arg1, strlen( arg1 ) );

   /*
    * Next, prepare the result string. It is common practice in Rexx that the
    * caller provides a buffer for the callee to return data. This isn't
    * a must. It is conforming to the definition to do a
    * "MAKERXSTRING( Result, NULL, 0 );", but this has the disadvantage that
    * the callee needs to allocate a buffer in all cases. This is very
    * time expensive. Passing a buffer from the current stack is very fast
    * and inexpensive.
    * You can choose the size by your own. RXAUTOBUFLEN (defined to 256) is
    * the most used value, but the size is not a "must".
    * The buffer itself will be used or not, replaced or not, but it is
    * never freed by the interpreter. On the other hand, if the interpreter
    * needs or wants to replace the buffer, the new buffer should be freed
    * by RexxFreeMemory, since the allocation is done by the interpreter
    * using RexxAllocateMemory.
    */
   MAKERXSTRING( Result, buf, sizeof( buf ) );

   rc = RexxCallBack( function,    /* ProcedureName */
                      1,           /* ArgCount */
                      ArgList,
                      &ReturnCode,
                      &Result );
   /*
    * The ReturnCode will be either 0 or the value of Result, if Result is a
    * valid number.
    * The return value of the function is one of RX_CB_BADP, RX_CB_NOTSTARTED,
    * RX_CB_TOOMANYP, RX_CB_BADN. Something's gone wrong in case of a
    * value != RX_CB_OK.
    */

   printf( "callback of `%s' returns %lu",
           function,
           rc );
   if ( rc == RX_CB_OK )
      printf( ", ReturnCode=%hd, Result=%.*s\n",
              ReturnCode,
              (int) RXSTRLEN( Result ), RXSTRPTR( Result ) );
   else
      printf( "\n" );

   /*
    * As mentioned above, the content of Result must be checked for a changed
    * buffer value. We have to free it, if it has changed.
    */
   if ( !RXNULLSTRING( Result ) &&
        ( (char *) (RXSTRPTR( Result )) != buf ) )
      RexxFreeMemory( RXSTRPTR( Result ) );
}

/*
 * Have a look at "RexxFunctionHandler" in the header file.
 * Have a look at step4.c for a full description of the arguments.
 *
 * This "function" allows the use in a CALL instruction only, it returns
 * the NULL string.
 */
static APIRET APIENTRY function( PCSZ name,
                                 ULONG argc,
                                 PRXSTRING argv,
                                 PCSZ queuename,
                                 PRXSTRING returnstring )
{
   clock_t end;

   (argc = argc);
   (argv = argv);
   (queuename = queuename);
   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return non-zero in this case.
    */
   if ( strcmp( name, "GUIEMULATOR" ) != 0 )
      return 1;

   /*
    * We do some calls back into the script to feed the script with some
    * informations.
    * The script itself needs some "entry points", function names, which must
    * be named "OnMouseMove" and "OnMouseClick" in this example. The names
    * are choosen either by the helper function or by the user as arguments
    * passed to the function.
    */

   DoCallback("OnMouseMove", "10,1" );
   DoCallback("OnMouseMove", "10,2" );

   /*
    * We do a very expensive sleep here. But this works for all machines.
    */
   end = clock() + (clock_t) CLOCKS_PER_SEC / 2;
   while ( clock() < end )
      ;

   DoCallback("OnMouseMove", "9,3" );
   DoCallback("OnMouseClick", "right" );
   DoCallback("unknown_function", "Hello" );

   MAKERXSTRING( *returnstring, NULL, 0 );
   return 0;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "right='r'\r\n"
                        "call GUIemulator\r\n"
                        "say 'Back in the script'\r\n"
                        "return 0\r\n"
                        "OnMouseMove:\r\n"
                        "   say 'callback: mouse moved to' arg(1)\r\n"
                        "   mouseIsAt = arg(1)\r\n"
                        "   return 1\r\n"
                        "OnMouseClick:\r\n"
                        "   say 'callback: mouse clicked with' arg(1) 'button'\r\n"
                        "   say 'buffered position is' mouseIsAt\r\n"
                        "   return 'WhatYouLike'\r\n";
   RXSTRING instore[2];


   /*
    * Before you can use a Rexx function hook you have to register the
    * function.
    * We register a function handler in the current executable in opposite to
    * a handler in an external library.
    *
    * ONLY USE RexxRegisterFunctionExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    */
   retval = RexxRegisterFunctionExe( "GUIemulator",
                                     (PFN) function);
   if ( retval != RXFUNC_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXFUNC_??? values
       * for possible errors.
       */
      printf( "Registering of the function 'GUIemulator' ended with code %u\n",
              (unsigned) retval );
      return 1;
   }

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       NULL,         /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   /*
    * We have to deregister the function handler. This isn't really needed
    * before the program's end but provided as an example for proper
    * programming.
    */
   RexxDeregisterFunction( "GUIemulator" );

   return (int) ( retval ? retval : rc );
}
