/*
 * This example lets a user's program stop the execution when running a Rexx
 * script within the program's environment. The script will get a notification
 * if a HALT handler is installed.
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterFunctionExe, RexxDeregisterFunction,
 *                 RexxSetHalt
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING
 * You should have read: step2-4.c, step4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN FUNCTION HANDLER SET IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Function" and "Asynchroneous Request
 * Interface" stuff.
 */
#define INCL_RXFUNC
#define INCL_RXARI
#include "rexx_header.h"
#include <stdio.h>
#include <string.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * Have a look at "RexxFunctionHandler" in the header file.
 * Have a look at step4.c for a full description of the arguments.
 *
 * This "function" allows the use in a CALL instruction only, it returns
 * the NULL string.
 */
static APIRET APIENTRY function( PCSZ name,
                                 ULONG argc,
                                 PRXSTRING argv,
                                 PCSZ queuename,
                                 PRXSTRING returnstring )
{
   static int counter = 0;

   (argc = argc);
   (argv = argv);
   (queuename = queuename);
   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return non-zero in this case.
    */
   if ( strcmp( name, "DIEAFTER7CALLS" ) != 0 )
      return 1;

   counter++;
   printf( "function `%s' entered%s\n",
           name,
           ( counter >= 7 ) ? ", will raise a HALT condition" : "" );

   /*
    * RexxSetHalt needs two parameters: PID and TID. The first one is the
    * process id, the second one the thread id. IBM's API specifies that a
    * TID of 0 will let a HALT condition raise in all threads of the mentioned
    * process.
    *
    * Regina doesn't do this. It may cause very severe errors. The only thread
    * we can signal is the current thread. Indeed, Regina ignores both
    * parameters and raises the HALT condition in every case.
    * Portable code should set PID and TID to the values of the current
    * thread, and it should check the return value.
    */
   if ( counter >= 7 )
   {
      RexxSetHalt( 0,
                   0 );
      printf( "RexxSetHalt called\n" );
   }

   MAKERXSTRING( *returnstring, NULL, 0 );
   return 0;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "signal on halt\r\n"
                        "do 8\r\n"
                        "   call DieAfter7Calls\r\n"
                        "   end\r\n"
                        "say 'HALT handler not entered'\r\n"
                        "return 1\r\n"
                        "HALT:\r\n"
                        "   say 'HALT handler entered'\r\n"
                        "   exit 1\r\n";
   RXSTRING instore[2];


   /*
    * Before you can use a Rexx function hook you have to register the
    * function.
    * We register a function handler in the current executable in opposite to
    * a handler in an external library.
    *
    * ONLY USE RexxRegisterFunctionExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    */
   retval = RexxRegisterFunctionExe( "DieAfter7Calls",
                                     (PFN) function);
   if ( retval != RXFUNC_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXFUNC_??? values
       * for possible errors.
       */
      printf( "Registering of the function 'DieAfter7Calls' ended with code "
                                                                        "%u\n",
              (unsigned) retval );
      return 1;
   }

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       NULL,         /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   /*
    * We have to deregister the function handler. This isn't really needed
    * before the program's end but provided as an example for proper
    * programming.
    */
   RexxDeregisterFunction( "DieAfter7Calls" );

   return (int) ( retval ? retval : rc );
}
