/*
 * This example lets a user's program extend the external command set when
 * running a Rexx script within the program's environment.
 * You'll get a slower performance using the technique of step3-8.c in most
 * cases.
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterSubcomExe, RexxDeregisterSubcom, RexxQuerySubcom
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING
 * You should have read: step2-4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN COMMAND HANDLER SET IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Sub-Command" stuff.
 */
#define INCL_RXSUBCOM
#include "rexx_header.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * setReturn copies the content into retval. A fresh buffer is allocated if
 * retval's buffer is too small. A terminating '\0' is silently appended.
 *
 * The Rexx interpreter has a return buffer of RXAUTOBUFLEN allocated
 * usually. The value is 256 in most cases.
 * 1) We don't rely on this.
 * 2) We are not allowed to free the string if it is too small. It is the
 *    property of the interpreter. It will check for a "lost" string by its
 *    own.
 * 3) Using or replacing the return string is allowed.
 * 4) For good practice we append a not counted ASCII terminator.
 * Regina uses 256 bytes, provides a line terminator but doesn't expect it.
 *
 * Allocating memory should be done with RexxAllocateMemory. This function
 * can be used with Regina and some other interpreters. It is not a known
 * function in IBM's API.
 */
static void setReturn( PRXSTRING retval, const char *content )
{
   char *p;
   int len = strlen( content );

   if ( RXNULLSTRING( *retval ) ||
        ( RXSTRLEN( *retval ) <= len ) )
      p = RexxAllocateMemory( len + 1 );
   else
      p = RXSTRPTR( *retval );
   if ( p != NULL )
      memcpy( p, content, len );
   else
   {
      /*
       * One can set the flags value to FAILURE in case of an error.
       */
   }
   MAKERXSTRING( *retval, p, len );
}

/*
 * isExit checks the content of the string for the case-insensitive
 * word "exit" as the only word.
 */
static int isExit( PRXSTRING string )
{
   const char *p = RXSTRPTR( *string );
   int len = (int) RXSTRLEN( *string );

   len -= 3; /* Allows us to compare for the whole "exit" at once */

   while ( len > 0 )
   {
      if ( ( toupper( p[0] ) == 'E' ) &&
           ( toupper( p[1] ) == 'X' ) &&
           ( toupper( p[2] ) == 'I' ) &&
           ( toupper( p[3] ) == 'T' ) )
      {
         len -= 4;
         p += 4;
         while ( len-- > 0 )
            if ( !isspace( *p ) )
               return 0;
         return 1;
      }
      if ( !isspace( p[0] ) )
         return 0;
      len--;
      p++;
   }
   return 0;
}

/*
 * Have a look at "RexxSubcomHandler" in the header file.
 * A subcom handler will be invoked by the Rexx interpreter if the command
 * is detected and the current ADDRESS environment is set to the subcom's
 * ADDRESS name.
 * This function receives the command which should be executed as the first
 * argument.
 * The next argument contains the flags-field which must be set to either
 * RXSUBCOM_OK, RXSUBCOM_ERROR or RXSUBCOM_FAILURE. The ERROR and FAILURE
 * values will raise the condition of the same name on return.
 * The last argument, returnstring,  must be set, too on success. The assigned
 * value will be assigned to the RC variable. If the assigned string is the
 * NULL-string ( RXSTRPTR( *returnstring ) == NULL ), then the interpreter
 * will use the string "0" for RC.
 * The return value of this function is currently ignored under Regina.
 * You should return 0 in all cases.
 *
 * USING DIFFERENT SUBCOM ENTRY POINTS IMPROVES THE PERFORMANCE. THIS
 * EXAMPLE DOESN'T TAKE ADVANTAGE OF IT.
 *
 * NEVER FORGET "APIENTRY" as a modifier to the function. Your stack becomes
 * confused otherwise and you'll have to understand why the return of the
 * caller of this function does a return to a random stack place.
 */
static APIRET APIENTRY subcom( PRXSTRING command,
                               PUSHORT flags,
                               PRXSTRING returnstring )
{
   static FILE *out = NULL;
   static char *cmd = NULL;

   if ( out == NULL )
   {
      printf( "Opening a new interactive shell under control of REXX\n" );
#if defined(WIN32) || defined(DOS) || defined(OS2)
      cmd = getenv( "COMSPEC" );
#else
      cmd = getenv( "SHELL" );
#endif
      if ( ( out = popen( cmd, "w" ) ) == NULL )
      {
         printf( "Error invoking an interpreter, will raise FAILURE\n" );
         /*
          * FAILURE should be raised if it isn't possible to execute something
          * at all and will never succeed.
          */
         *flags = RXSUBCOM_FAILURE;
         setReturn( returnstring, "-1" );
         return 0;
      }
#ifdef SIGPIPE
      signal( SIGPIPE, SIG_IGN );
      /*
       * The signal will not be reset in this example as it should in real
       * life.
       */
#endif
   }

   /*
    * We check for EXIT by ourself, otherwise an EXIT passed to the inferior
    * interpreter will raise a "broken pipe" signal probably on the next
    * try to write something. That's too late.
    * The application can always fake us by something like "echo x; exit".
    */
   if ( isExit( command ) )
   {
      fclose( out );
      out = NULL;
      printf( "connection to inferior interpreter cutted\n" );
      *flags = RXSUBCOM_OK;
      setReturn( returnstring, "1" );
      return 0;
   }

   if ( ( fprintf( out, "%.*s\n",
                        (int) RXSTRLEN( *command ),
                        RXSTRPTR( *command ) ) < 0 ) ||
        ( fflush( out ) == EOF ) )
   {
      fclose( out );
      out = NULL;
      printf( "connection to inferior interpreter broken, will raise ERROR\n" );
      /*
       * ERROR should be raised if a command, better THIS command, couldn't
       * be executed without an error.
       * Indeed, we should have to check the inferior's error channel, too,
       * but this is just a stupid example.
       */
      *flags = RXSUBCOM_ERROR;
      setReturn( returnstring, "1" );
      return 0;
   }

   printf( "Type `EXIT' to stop passing commands from the script to %s\n",
           cmd );
   *flags = RXSUBCOM_OK;
   setReturn( returnstring, "0" );
   return 0;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   USHORT flag;
   char buf[8];
   static char *macro = "'echo \"Foo is the current environment\"'\r\n"
                        "address ISYSTEM\r\n"
                        /* A directed ADDRESS command is always possible */
                        "address SYSTEM 'echo \"Enter interactive commands\"'\r\n"
                        /* Everything else is executed directly in ISYSTEM */
                        "'echo \"Stop with EXIT\"'\r\n"
                        "do until RC \\= 0\r\n"
                        "   parse pull line\r\n"
                        "   line\r\n"
                        "   end\r\n"
                        "say 'Back in REXX, finishing'\r\n";
   RXSTRING instore[2];


   /*
    * Before you can use a Rexx subcom handler you have to register the
    * handler.
    *
    * Rexx provides two different versions for registering. The appropriate one
    * in this situation and the simpler one is RexxRegisterSubcomExe. The other
    * one is RexxRegisterSubcomDll. Like all other registering functions of
    * Rexx, the "Exe" version fetches just a function pointer, while the
    * "Dll" version needs the name of a shared library (aka DLL) and the name
    * of an exported function/procedure of the shared library.
    *
    * The "Dll" version links the shared library, fetches the procedure entry
    * of the given function name and behaves like the RexxRegisterSubcomExe
    * with the fetched procedure entry. The "Dll" version shall be used in all
    * cases where external libraries are used. The Rexx interpreter knows what
    * to do in cases where different registrations have been done and a
    * deregistration happens.
    *
    * One scenario where these considerations take effect is this:
    * A Rexx support package is registered by a call to RxFuncAdd. The function
    * of the shared library registers an subcom handler for some purpose. It
    * does this by RexxRegisterSubcomExe. This will work for the first. But
    * supposed the function is dropped, e.g. by RxFuncDrop, the Rexx
    * interpreter may decide to unlink the DLL from its working set of
    * libraries. The next try to use the subcom handler will let the Rexx
    * interpreter jump into outer space and will hopefully be killed the
    * operating system due to an illegal use of memory.
    *
    * ONLY USE RexxRegisterSubcomExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    *
    * The other parameters of RexxRegisterSubcom... are the name of the address
    * environment for Rexx and a private area which can be used for magic
    * things. The name is rather useful, since it is possible for the script
    * to use the ADDRESS instruction to select a specific evvironment.
    * The name is used case-sensitive.
    *
    * The user area is rarely useful in most cases. A value of NULL is allowed.
    * If it is set, a 8 byte buffer is copied from that address and associated
    * with the exit handler. Regina doesn't inspect the buffer content.
    *
    * RexxRegisterSubcomDll isn't supported by Regina including Version 3.2.
    */
   retval = RexxRegisterSubcomExe( "ISYSTEM",
                                   (PFN) subcom,
                                   NULL ); /* No user area, else 8 byte */
   if ( retval != RXSUBCOM_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXSUBCOM_???
       * values for possible errors.
       */
      printf( "Registering of the subcom 'ISYSTEM' ended with code %u\n",
              (unsigned) retval );
      return 1;
   }

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       NULL,         /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   if ( RexxQuerySubcom( "ISYSTEM",
                         NULL,      /* Look for "Exe"-registered handlers */
                         &flag,
                         buf /* may be NULL if uninteresting*/) != RXSUBCOM_OK )
   {
      printf( "Error: subcom `ISYSTEM' is no longer registered.\n" );
      if ( flag != 0 )
         printf( "RexxQuerySubcom doesn't honour the `flag' argument "
                                                               "properly.\n" );
   }
   else if ( flag != RXSUBCOM_ISREG )
      printf( "RexxQuerySubcom doesn't honour the `flag' argument "
                                                               "properly.\n" );

   if ( RexxQuerySubcom( "not registered",
                         NULL,      /* Look for "Exe"-registered handlers */
                         &flag,
                         NULL ) == RXSUBCOM_NOTREG )
   {
      if ( flag != 0 )
         printf( "RexxQuerySubcom doesn't honour the `flag' argument "
                                                               "properly.\n" );
   }
   else
      printf( "RexxQuerySubcom returns nonsense.\n" );

   /*
    * We have to deregister the subcom handler. This isn't really needed
    * before the program's end but provided as an example for proper
    * programming.
    * The first parameter is the name of an address environment while the
    * second one, ModuleName, is ignored by Regina currently. It may be NULL
    * for the current executable or the name of a shared library (as given when
    * registering) to limit the search to just that library.
    * Note that a duplicate registration with the same name and different
    * modules may happen and is allowed. When deregistering, the Exe version
    * is searched first for the handler. Regina uses just one list for all
    * handlers.
    */
   RexxDeregisterSubcom( "ISYSTEM",
                         NULL ); /* Deregister an "Exe"-Subcom */

   return (int) ( retval ? retval : rc );
}
