/*
 * This example lets a user's program extend the function set when
 * running a Rexx script within the program's environment.
 * You'll get a slower performance using the technique of step3-7.c in most
 * cases.
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterFunctionExe, RexxDeregisterFunction,
 *                 RexxQueryFunction
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING
 * You should have read: step2-4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN FUNCTION HANDLER SET IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Function" stuff.
 */
#define INCL_RXFUNC
#include "rexx_header.h"
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * This function is an extension of the builtin function POS. It operates
 * case-insensitiv.
 * The strings in the arguments don't have to be 0-terminated.
 */
static int caseIgnorePos( const char *needle,
                          int needlelen,
                          const char *haystack,
                          int haystacklen )
{
   int i, j;
   /*
    * Regina uses a Boyer-Moore algorithm, we use a really stupid one here.
    */
   haystacklen -= needlelen - 1;

   for ( i = 0; i < haystacklen; i++ )
   {
      for ( j = 0; j < needlelen; j++ )
      {
         if ( toupper( haystack[i] ) != toupper( needle[j] ) )
            break;
      }
      if ( j >= needlelen )
         return i + 1;
   }
   return 0;
}

/*
 * Have a look at "RexxFunctionHandler" in the header file.
 * A function handler will be invoked by the Rexx interpreter if the name of
 * the function appears in a "function" context and the function isn't known
 * as an internal function.
 * The function receives the name under which it is invoked and the arguments.
 * Next it receives the name of the current queue. This is helpful for
 * accessing the default queue during the processing of the function.
 * The function may return a complex string as its return value in the last
 * variable.
 * The return value of the function is either zero for success or non-zero
 * for a request of raising a generic syntax error of 40.
 *
 * argv may contain arguments with some of the elements' strptr set to NULL.
 * This indicates omitted values as in func(a,,,b).
 *
 * USING DIFFERENT FUNCTION ENTRY POINTS FOR EACH FUNCTION IMPROVES THE
 * PERFORMANCE.
 * THIS EXAMPLE DOESN'T TAKE ADVANTAGE OF IT.
 *
 * NEVER FORGET "APIENTRY" as a modifier to the function. Your stack becomes
 * confused otherwise and you'll have to understand why the return of the
 * caller of this function does a return to a random stack place.
 */
static APIRET APIENTRY function( PCSZ name,
                                 ULONG argc,
                                 PRXSTRING argv,
                                 PCSZ queuename,
                                 PRXSTRING returnstring )
{
   enum { isPos, isSink, unknown } function = unknown;
   char buf[40], *p;
   int h;

   (queuename = queuename);

   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return non-zero in this case.
    */
   if ( strcmp( name, "CIPOS" ) == 0 )
      function = isPos;
   if ( strcmp( name, "SINK" ) == 0 )
      function = isSink;

   if ( function == unknown )
      return 1;

   printf( "function %s entered\n", name );

   if ( function == isSink )
   {
      /*
       * Sink doesn't return a value at all. You can use "call sink whatever"
       * but you can't use it as a function like "x = sink(whatever)"
       * For this behaviour set the return value's pointer to NULL.
       */
      MAKERXSTRING( *returnstring, NULL, 0 );
      return 0;
   }

   /*
    * Compute the value of exp(arg(0)) if all parameters fit.
    */
   if ( ( argc != 2 ) ||
        ( RXSTRLEN( argv[0] ) == 0 ) ||
        ( RXSTRLEN( argv[1] ) == 0 ) )
   {
      printf("(returning an error)\n");
      return 1;
   }

   sprintf( buf, "%d", caseIgnorePos( RXSTRPTR( argv[0] ),
                                      (int) RXSTRLEN( argv[0] ),
                                      RXSTRPTR( argv[1] ),
                                      (int) RXSTRLEN( argv[1] ) ) );

   /*
    * The Rexx interpreter has a return buffer of RXAUTOBUFLEN allocated
    * usually. The value is 256 in most cases.
    * 1) We don't rely on this.
    * 2) We are not allowed to free the string if it is too small. It is the
    *    property of the interpreter. It will check for a "lost" string by its
    *    own.
    * 3) Using or replacing the return string is allowed.
    * 4) For good practice we append a not counted ASCII terminator.
    * Regina uses 256 bytes, provides a line terminator but doesn't expect it.
    *
    * Allocating memory should be done with RexxAllocateMemory. This function
    * can be used with Regina and some other interpreters. It is not a known
    * function in IBM's API.
    */
   h = strlen( buf );
   if ( h < RXSTRLEN( *returnstring ) )
      p = RXSTRPTR( *returnstring );
   else
   {
      p = RexxAllocateMemory( h + 1 );
      if ( p == NULL )
         return 1;
   }
   memcpy( p, buf, h );
   p[h] = '\0';
   MAKERXSTRING( *returnstring, p, h );
   return 0;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "say \"cipos('A', 'Salad') =\" cipos('A', 'Salad')\r\n"
                        "call sink 1, 2, 3\r\n"
                        "say 'checking the error conditions...'\r\n"
                        "signal on syntax name s1\r\n"
                        "call cipos 1, 2, 3\r\n"
                        "say 'oops, cipos() allows three arguments'\r\n"
                        "s1:\r\n"
                        "signal on syntax name s2\r\n"
                        "h = sink( 1, 2, 3)\r\n"
                        "say 'oops, sink() can be used as a function'\r\n"
                        "s2:\r\n";
   RXSTRING instore[2];


   /*
    * Before you can use a Rexx function handler you have to register the
    * handler.
    *
    * Rexx provides two different versions for registering. The appropriate one
    * in this situation and the simpler one is RexxRegisterFunctionExe. The
    * other one is RexxRegisterFunctionDll. Like all other registering
    * functions of Rexx, the "Exe" version fetches just a function pointer,
    * while the "Dll" version needs the name of a shared library (aka DLL) and
    * the name of an exported function/procedure of the shared library.
    *
    * The "Dll" version links the shared library, fetches the procedure entry
    * of the given function name and behaves like the RexxRegisterFunctionExe
    * with the fetched procedure entry. The "Dll" version shall be used in all
    * cases where external libraries are used. The Rexx interpreter knows what
    * to do in cases where different registrations have been done and a
    * deregistration happens.
    *
    * One scenario where these considerations take effect is this:
    * A Rexx support package is registered by a call to RxFuncAdd. The function
    * of the shared library registers an Exit handler for some purpose. It does
    * this by RexxRegisterFunctionExe. This will work for the first. But
    * supposed the function is dropped, e.g. by RxFuncDrop, the Rexx
    * interpreter may decide to unlink the DLL from its working set of
    * libraries. The next try to use the function handler will let the Rexx
    * interpreter jump into outer space and will hopefully be killed the
    * operating system due to an illegal use of memory.
    *
    * ONLY USE RexxRegisterFunctionExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    *
    * The other parameter of RexxRegisterFunction... is the name of the hook.
    * The name is used case-insensitive.
    */

   retval = RexxRegisterFunctionExe( "cipos",
                                     (PFN) function);
   if ( retval != RXFUNC_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXFUNC_??? values
       * for possible errors.
       */
      printf( "Registering of the function 'cipos' ended with code %u\n",
              (unsigned) retval );
   }
   retval = RexxRegisterFunctionExe( "sink",
                                     (PFN) function);
   if ( retval != RXFUNC_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXFUNC_??? values
       * for possible errors.
       */
      printf( "Registering of the function 'sink' ended with code %u\n",
              (unsigned) retval );
   }

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       NULL,         /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   if ( RexxQueryFunction( "CIPos" ) != RXFUNC_OK ) /* else RXFUNC_NOTREG */
      printf( "Error: function `cipos' is no longer registered.\n" );

   if ( RexxQueryFunction( "not registered" ) != RXFUNC_NOTREG )
      printf( "Error: RexxQueryFunction returns nonsense.\n" );

   /*
    * We have to deregister the function handlers. This isn't really needed
    * before the program's end but provided as an example for proper
    * programming.
    */
   RexxDeregisterFunction( "cipos" );
   RexxDeregisterFunction( "sink" );

   return (int) ( retval ? retval : rc );
}
