/*
 * This example lets a user's program intercept every external command when
 * running a Rexx script within the program's environment.
 * You'll get a better performance using the technique of step5.c in most
 * cases.
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterExitExe, RexxDeregisterExit
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING, RXSYSEXIT, PEXIT, RXCMDHST_PARM
 * You should have read: step3-1.c, step2-4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN EXIT HANDLER SET IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Exit" stuff.
 */
#define INCL_RXSYSEXIT
#include "rexx_header.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * setReturn copies the content into retval. A fresh buffer is allocated if
 * retval's buffer is too small. A terminating '\0' is silently appended.
 * The return value is RXEXIT_HANDLED on success or RXEXIT_RAISE_ERROR on
 * error.
 *
 * The Rexx interpreter has a return buffer of RXAUTOBUFLEN allocated
 * usually. The value is 256 in most cases.
 * 1) We don't rely on this.
 * 2) We are not allowed to free the string if it is too small. It is the
 *    property of the interpreter. It will check for a "lost" string by its
 *    own.
 * 3) Using or replacing the return string is allowed.
 * 4) For good practice we append a not counted ASCII terminator.
 * Regina uses 256 bytes, provides a line terminator but doesn't expect it.
 *
 * Allocating memory should be done with RexxAllocateMemory. This function
 * can be used with Regina and some other interpreters. It is not a known
 * function in IBM's API.
 */
static ULONG setReturn( PRXSTRING retval, const char *content )
{
   char *p;
   int len = strlen( content );

   if ( RXNULLSTRING( *retval ) ||
        ( RXSTRLEN( *retval ) <= len ) )
      p = RexxAllocateMemory( len + 1 );
   else
      p = RXSTRPTR( *retval );

   if ( p == NULL )
      return RXEXIT_RAISE_ERROR;

   memcpy( p, content, len );
   MAKERXSTRING( *retval, p, len );
   return RXEXIT_HANDLED;
}

/*
 * isExit checks the content of the string for the case-insensitive
 * word "exit" as the only word.
 */
static int isExit( PRXSTRING string )
{
   const char *p = RXSTRPTR( *string );
   int len = (int) RXSTRLEN( *string );

   len -= 3; /* Allows us to compare for the whole "exit" at once */

   while ( len > 0 )
   {
      if ( ( toupper( p[0] ) == 'E' ) &&
           ( toupper( p[1] ) == 'X' ) &&
           ( toupper( p[2] ) == 'I' ) &&
           ( toupper( p[3] ) == 'T' ) )
      {
         len -= 4;
         p += 4;
         while ( len-- > 0 )
            if ( !isspace( *p ) )
               return 0;
         return 1;
      }
      if ( !isspace( p[0] ) )
         return 0;
      len--;
      p++;
   }
   return 0;
}

/*
 * Have a look at "RexxExitHandler" in the header file.
 * An exit handler will get some IO requests of the Rexx interpreter. It may
 * or may not handle it.
 * main_code is one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC, RXINI, RXTER,
 * RXDBG, RXENV currently.
 * sub_code are different. Look at the define'd values in the header file
 * starting with the name of the main_code, e.g. RXCMDHST.
 * data is a generic placeholder for the true data structure depending on
 * the main_code. We use here RXCMDHST_PARM and have to cast data to such a
 * structure.
 * NEVER FORGET "APIENTRY" as a modifier to the function. Your stack becomes
 * confused otherwise and you'll have to understand why the return of the
 * caller of this function does a return to a random stack place.
 */
static LONG APIENTRY interceptor( LONG main_code,
                                  LONG sub_code,
                                  PEXIT data )
{
   char *p;
   int h;
   static FILE *out = NULL;
   static char *cmd = NULL;
   RXCMDHST_PARM *arg = (RXCMDHST_PARM *) data;

   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return RXEXIT_RAISE_ERROR or RXEXIT_NOT_HANDLED in this case.
    */
   if ( main_code != RXCMD )
      return RXEXIT_RAISE_ERROR;

   /*
    * Every sub_code must be expected and RXEXIT_NOT_HANDLED should be returned
    * for every sub_code that shouldn't be handled.
    * Anything other than the function execution is treated as an error here.
    * A more sophisticated logic should be implemented for the different cases.
    */
   if ( sub_code != RXCMDHST )
      return RXEXIT_NOT_HANDLED;

   /*
    * Command execution. We have several parameters. I indicates an input
    * parameter and must not be changed. O indicates an output parameter which
    * may be changed.
    * O arg->rxcmd_flags.rxfcfail  The interpreter should raise a FAILURE
    *                              condition which may be trapped by CALL or
    *                              SIGNAL instruction.
    * O arg->rxcmd_flags.rxfcerr   The interpreter should raise an ERROR
    *                              condition which may be trapped by CALL or
    *                              SIGNAL instruction.
    * I arg->rxcmd_address         The name of the address environment. Its
    *                              length is arg->rxcmd_addressl. The name is
    *                              0-terminated in Regina and in many other
    *                              interpreters.
    * I arg->rxfnc_addressl        Valid characters in arg->rxcmd_address.
    * I arg->rxcmd_dll             The name of the current shared library aka
    *                              DLL. Its length is arg->rxcmd_dll_len. The
    *                              name is 0-terminated in Regina and in many
    *                              other interpreters.
    * I arg->rxcmd_dll_len         Valid characters in arg->rxcmd_dll.
    * I arg->rxcmd_command         The command which has to be processed.
    *                              The command string is 0-terminated in Regina
    *                              and in many other interpreters.
    * O arg->rxcmd_retc            The value which should be assigned to the
    *                              special variable RC after the execution.
    *                              A default buffer of 256 characters
    *                              (RXAUTOBUFLEN) is provided which may be
    *                              used. The buffer's content pointer must not
    *                              be freed. The buffer's content pointer may
    *                              be replaced by a new buffer which is freed
    *                              by the interpreter immediately after return.
    *                              Such a buffer must be allocated by
    *                              RexxAllocateMemory. This function is not
    *                              portable, e.g. it is missing in IBM's API.
    *
    * In most cases it is much faster for the interpreter to execute a command
    * registered as an external command by RexxRegisterSubcom... as in
    * step5.c
    * This method is good for cheating arguments or return values of other
    * command packages since it is checked/executed before any registered
    * external command interpreter is exeuted.
    */

   p = arg->rxcmd_address;
   h = arg->rxcmd_addressl;
   if ( ( h != 7 ) || ( memcmp( p, "ISYSTEM", h ) != 0 ) )
   {
      printf( "address environment `%.*s' not handled.\n", h, p );
      return RXEXIT_NOT_HANDLED;
   }

   if ( out == NULL )
   {
      printf( "Opening a new interactive shell under control of REXX\n" );
#if defined(WIN32) || defined(DOS) || defined(OS2)
      cmd = getenv( "COMSPEC" );
#else
      cmd = getenv( "SHELL" );
#endif
      if ( ( out = popen( cmd, "w" ) ) == NULL )
      {
         printf( "Error invoking an interpreter, will raise FAILURE\n" );
         /*
          * FAILURE should be raised if it isn't possible to execute something
          * at all and will never succeed.
          */
         arg->rxcmd_flags.rxfcfail = 1;
         return setReturn( &arg->rxcmd_retc, "-1" );
      }
#ifdef SIGPIPE
      signal( SIGPIPE, SIG_IGN );
      /*
       * The signal will not be reset in this example as it should in real
       * life.
       */
#endif
   }

   /*
    * We check for EXIT by ourself, otherwise an EXIT passed to the inferior
    * interpreter will raise a "broken pipe" signal probably on the next
    * try to write something. That's too late.
    * The application can always fake us by something like "echo x; exit".
    */
   if ( isExit( &arg->rxcmd_command ) )
   {
      fclose( out );
      out = NULL;
      printf( "connection to inferior interpreter cutted\n" );
      return setReturn( &arg->rxcmd_retc, "1" );
   }

   if ( ( fprintf( out, "%.*s\n",
                        (int) RXSTRLEN( arg->rxcmd_command ),
                        RXSTRPTR( arg->rxcmd_command ) ) < 0 ) ||
        ( fflush( out ) == EOF ) )
   {
      fclose( out );
      out = NULL;
      printf( "connection to inferior interpreter broken, will raise ERROR\n" );
      /*
       * ERROR should be raised if a command, better THIS command, couldn't
       * be executed without an error.
       * Indeed, we should have to check the inferior's error channel, too,
       * but this is just a stupid example.
       */
      arg->rxcmd_flags.rxfcerr = 1;
      return setReturn( &arg->rxcmd_retc, "1" );
   }

   printf( "Type `EXIT' to stop passing commands from the script to %s\n",
           cmd );
   return setReturn( &arg->rxcmd_retc, "0" );
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "'echo \"Foo is the current environment\"'\r\n"
                        "address ISYSTEM\r\n"
                        /* A directed ADDRESS command is always possible */
                        "address SYSTEM 'echo \"Enter interactive commands\"'\r\n"
                        /* Everything else is executed directly in ISYSTEM */
                        "'echo \"Stop with EXIT\"'\r\n"
                        "do until RC \\= 0\r\n"
                        "   parse pull line\r\n"
                        "   line\r\n"
                        "   end\r\n"
                        "say 'Back in REXX, finishing'\r\n";
   RXSTRING instore[2];
   RXSYSEXIT exits[2];

   /*
    * Before you can use a Rexx exit hook you have to register the hook.
    * We register an exit handler in the current executable in opposite to
    * a handler in an external library.
    *
    * ONLY USE RexxRegisterExitExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    */

   retval = RexxRegisterExitExe( "myExitHandler",
                                 (PFN) interceptor,
                                 NULL );
   if ( retval != RXEXIT_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXEXIT_??? values
       * for possible errors.
       */
      printf( "Registering of the exit hook ended with code %u\n",
              (unsigned) retval );
   }

   /*
    * Setup the list of Exit hooks. The last one must contain the value
    * RXENDLST as the sysexit_code. sysexit_name's value is irrelevant in this
    * case.
    * Every "useful" entry should have both a registered exit handler's name
    * as sysexit_name and one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC,
    * RXINI, RXTER, RXDBG, RXENV as sysexit_code.
    */
   exits[0].sysexit_name = "myExitHandler";
   exits[0].sysexit_code = RXCMD;
   exits[1].sysexit_code = RXENDLST;

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       exits,        /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   /*
    * We have to deregister the exit hook. This isn't really needed before
    * the program's end but provided as an example for proper programming.
    */
   RexxDeregisterExit( "myExitHandler", NULL );

   return (int) ( retval ? retval : rc );
}
