/*
 * This example lets a user's program intercept every external function call
 * when running a Rexx script within the program's environment.
 * You'll get a better performance using the technique of step4.c in most
 * cases.
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterExitExe, RexxDeregisterExit
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING, RXSYSEXIT, PEXIT, RXFNCCAL_PARM
 * You should have read: step3-1.c, step2-4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN EXIT HANDLER SET IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Exit" stuff.
 */
#define INCL_RXSYSEXIT
#include "rexx_header.h"
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * This function is an extension of the builtin function POS. It operates
 * case-insensitiv.
 * The strings in the arguments don't have to be 0-terminated.
 */
static int caseIgnorePos( const char *needle,
                          int needlelen,
                          const char *haystack,
                          int haystacklen )
{
   int i, j;
   /*
    * Regina uses a Boyer-Moore algorithm, we use a really stupid one here.
    */
   haystacklen -= needlelen - 1;

   for ( i = 0; i < haystacklen; i++ )
   {
      for ( j = 0; j < needlelen; j++ )
      {
         if ( toupper( haystack[i] ) != toupper( needle[j] ) )
            break;
      }
      if ( j >= needlelen )
         return i + 1;
   }
   return 0;
}

/*
 * Have a look at "RexxExitHandler" in the header file.
 * An exit handler will get some IO requests of the Rexx interpreter. It may
 * or may not handle it.
 * main_code is one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC, RXINI, RXTER,
 * RXDBG, RXENV currently.
 * sub_code are different. Look at the define'd values in the header file
 * starting with the name of the main_code, e.g. RXFNCCAL.
 * data is a generic placeholder for the true data structure depending on
 * the main_code. We use here RXFNCCAL_PARM and have to cast data to such a
 * structure.
 * NEVER FORGET "APIENTRY" as a modifier to the function. Your stack becomes
 * confused otherwise and you'll have to understand why the return of the
 * caller of this function does a return to a random stack place.
 */
static LONG APIENTRY interceptor( LONG main_code,
                                  LONG sub_code,
                                  PEXIT data )
{
   enum { isPos, isSink, unknown } function = unknown;
   char *p;
   int h;
   char buf[40];
   RXFNCCAL_PARM *arg = (RXFNCCAL_PARM *) data;

   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return RXEXIT_RAISE_ERROR or RXEXIT_NOT_HANDLED in this case.
    */
   if ( main_code != RXFNC )
      return RXEXIT_RAISE_ERROR;

   /*
    * Every sub_code must be expected and RXEXIT_NOT_HANDLED should be returned
    * for every sub_code that shouldn't be handled.
    * Anything other than the function execution is treated as an error here.
    * A more sophisticated logic should be implemented for the different cases.
    */
   if ( sub_code != RXFNCCAL )
      return RXEXIT_NOT_HANDLED;

   /*
    * Function execution. We have several parameters. I indicates an input
    * parameter and must not be changed. O indicates an output parameter which
    * may be changed.
    * O arg->rxfnc_flags.rxfferr   The interpreter should raise error 40 after
    *                              return.
    * O arg->rxfnc_flags.rxffnfnd  The interpreter should raise error 43 after
    *                              return.
    * I arg->rxfnc_flags.rxffsub   The current function is part of a "CALL"
    *                              instruction. The parameter
    *                              rxfnc_retc->strptr may be set to NULL. This
    *                              leads to a drop of the variable RESULT.
    *                              A function used in an evaluation expression
    *                              must return non-NULL.
    * I arg->rxfnc_name            The name of the function. Its length is
    *                              arg->rxfnc_namel. The name is 0-terminated
    *                              in Regina and in many other interpreters.
    * I arg->rxfnc_namel           Valid characters in arg->rxfnc_name.
    * I arg->rxfnc_que             The name of the current default queue. Its
    *                              length is arg->rxfnc_quel. The name is
    *                              0-terminated in Regina and in many other
    *                              interpreters.
    * I arg->rxfnc_quel            Valid characters in arg->rxfnc_que.
    * I arg->rxfnc_argc            Number of arguments in rxfnc_argv.
    * I arg->rxfnc_argv            Arguments to the function. The arguments'
    *                              strptr may be set to NULL to indicate
    *                              omitted values as in func(a,,,b).
    * O arg->rxfnc_retc            The return value of the function. It may
    *                              be set to NULL in case of a CALLed function
    *                              only. A default buffer of 256 characters
    *                              (RXAUTOBUFLEN) is provided which may be
    *                              used. The buffer's content pointer must not
    *                              be freed. The buffer's content pointer may
    *                              be replaced by a new buffer which is freed
    *                              by the interpreter immediately after return.
    *                              Such a buffer must be allocated by
    *                              RexxAllocateMemory. This function is not
    *                              portable, e.g. it is missing in IBM's API.
    *
    * In most cases it is much faster for the interpreter to execute a function
    * registered as an external function by RexxRegisterFunction... as in
    * step4.c
    * This method is good for cheating arguments or return values of other
    * function packages since it is checked/executed before any registered
    * function is exeuted.
    */

   p = arg->rxfnc_name;
   h = arg->rxfnc_namel;
   if ( ( h == 5 ) && ( memcmp( p, "CIPOS", h ) == 0 ) )
      function = isPos;
   if ( ( h == 4 ) && ( memcmp( p, "SINK", h ) == 0 ) )
      function = isSink;

   if ( function == unknown )
   {
      printf( "function %.*s not handled\n", h, p );
      return RXEXIT_NOT_HANDLED;
   }

   printf( "function %.*s entered%s\n",
           h, p,
           ( arg->rxfnc_flags.rxffsub ) ? " within a CALL instruction" : "" );

   if ( function == isSink )
   {
      /*
       * Sink doesn't return a value at all. You can use "call sink whatever"
       * but you can't use it as a function like "x = sink(whatever)"
       * For this behaviour set the return value's pointer to NULL.
       */
      MAKERXSTRING( arg->rxfnc_retc, NULL, 0 );
      return RXEXIT_HANDLED;
   }

   /*
    * Even after a "handled" event one may return "not handled" sometimes.
    * It depends on the function. A look over the parameter won't worry in
    * all cases and just inspecting data, e.g. for debugging purpose, is a
    * nice feature where RXEXIT_NOT_HANDLED is appropriate.
    *
    * RXEXIT_NOT_HANDLED instructs the interpreter to continue iterating
    * through the list of exit handlers for this main_code. If neither handler
    * returns RXEXIT_HANDLED, the default action happens.
    *
    * The handler should return RXEXIT_HANDLED in case of an error and set
    * the correct error indicator.
    */
   if ( ( arg->rxfnc_argc != 2 ) ||
        ( RXSTRLEN( arg->rxfnc_argv[0] ) == 0 ) ||
        ( RXSTRLEN( arg->rxfnc_argv[1] ) == 0 ) )
   {
      printf("(returning an error)\n");
      arg->rxfnc_flags.rxfferr = 1;
      MAKERXSTRING( arg->rxfnc_retc, NULL, 0 );
      return RXEXIT_HANDLED;
   }

   sprintf( buf, "%d", caseIgnorePos( RXSTRPTR( arg->rxfnc_argv[0] ),
                                      (int) RXSTRLEN( arg->rxfnc_argv[0] ),
                                      RXSTRPTR( arg->rxfnc_argv[1] ),
                                      (int) RXSTRLEN( arg->rxfnc_argv[1] ) ) );

   /*
    * The Rexx interpreter has a return buffer of RXAUTOBUFLEN allocated
    * usually. The value is 256 in most cases.
    * 1) We don't rely on this.
    * 2) We are not allowed to free the string if it is too small. It is the
    *    property of the interpreter. It will check for a "lost" string by its
    *    own.
    * 3) Using or replacing the return string is allowed.
    * 4) For good practice we append a not counted ASCII terminator.
    * Regina uses 256 byte, provides a line terminator but doesn't expect it.
    *
    * Allocating memory should be done with RexxAllocateMemory. This function
    * can be used with Regina and some other interpreters. It is not a known
    * function in IBM's API.
    */
   h = strlen( buf );
   if ( h < RXSTRLEN( arg->rxfnc_retc ) )
      p = RXSTRPTR( arg->rxfnc_retc );
   else
   {
      p = RexxAllocateMemory( h + 1 );
      if ( p == NULL )
         return RXEXIT_RAISE_ERROR;
   }
   memcpy( p, buf, h );
   p[h] = '\0';
   MAKERXSTRING( arg->rxfnc_retc, p, h );
   return RXEXIT_HANDLED;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "say \"POS('A', 'salad') =\" pos( 'A', 'salad' )\r\n"
                        "say \"CIPOS('A', 'salad') =\" cipos( 'A', 'salad' )\r\n"
                        "call sink A, B, C\r\n";
   RXSTRING instore[2];
   RXSYSEXIT exits[2];

   /*
    * Before you can use a Rexx exit hook you have to register the hook.
    * We register an exit handler in the current executable as opposed to
    * a handler in an external library.
    *
    * ONLY USE RexxRegisterExitExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    */

   retval = RexxRegisterExitExe( "myExitHandler",
                                 (PFN) interceptor,
                                 NULL );
   if ( retval != RXEXIT_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXEXIT_??? values
       * for possible errors.
       */
      printf( "Registering of the exit hook ended with code %u\n",
              (unsigned) retval );
   }

   /*
    * Setup the list of Exit hooks. The last one must contain the value
    * RXENDLST as the sysexit_code. sysexit_name's value is irrelevant in this
    * case.
    * Every "useful" entry should have both a registered exit handler's name
    * as sysexit_name and one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC,
    * RXINI, RXTER, RXDBG, RXENV as sysexit_code.
    */
   exits[0].sysexit_name = "myExitHandler";
   exits[0].sysexit_code = RXFNC;
   exits[1].sysexit_code = RXENDLST;

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       exits,        /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   /*
    * We have to deregister the exit hook. This isn't really needed before
    * the program's end but provided as an example for proper programming.
    */
   RexxDeregisterExit( "myExitHandler", NULL );

   return (int) ( retval ? retval : rc );
}
