/*
 * This example lets a user's program intercept the start and the end of the
 * interpretation when running a Rexx script within the program's environment.
 *
 * A typical usage initializes some variables on start or cleans up
 * external resources on exit.
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterExitExe, RexxDeregisterExit
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING, RXSYSEXIT
 * You should have read: step3-1.c, step2-4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN EXIT HANDLER SET IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Exit" stuff.
 */
#define INCL_RXSYSEXIT
#include "rexx_header.h"
#include <stdio.h>
#include <string.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * Have a look at "RexxExitHandler" in the header file.
 * These exit handlers will get some informational requests of the Rexx
 * interpreter.
 * main_code is one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC, RXINI, RXTER,
 * RXDBG, RXENV currently.
 * sub_code are different. Look at the define'd values in the header file
 * starting with the name of the main_code, e.g. RXINIEXT or RXTEREXT.
 * data is a generic placeholder for the true data structure depending on
 * the main_code. We don't use it, since the intercepted exit handlers don't
 * have any arguments.
 * NEVER FORGET "APIENTRY" as a modifier to the function. Your stack becomes
 * confused otherwise and you'll have to understand why the return of the
 * caller of this function does a return to a random stack place.
 */
static LONG APIENTRY interceptor( LONG main_code,
                                  LONG sub_code,
                                  PEXIT data )
{
   (data = data);

   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return RXEXIT_RAISE_ERROR or RXEXIT_NOT_HANDLED in this case.
    */
   if ( main_code == RXINI )
   {
      /*
       * Every sub_code must be expected and RXEXIT_NOT_HANDLED should be
       * returned for every sub_code that shouldn't be handled.
       */
      if ( sub_code == RXINIEXT )
      {
         printf( "A script is starting up.\n" );
         return RXEXIT_HANDLED;
      }
      return RXEXIT_NOT_HANDLED;
   }

   if ( main_code == RXTER )
   {
      if ( sub_code == RXTEREXT )
      {
         printf( "A script is stopping.\n" );
         return RXEXIT_HANDLED;
      }
      return RXEXIT_NOT_HANDLED;
   }

   return RXEXIT_RAISE_ERROR;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "say 'first instruction'\r\n"
                        "interpret 'say \"sub-interpreted\"'\r\n"
                        "say 'last instruction'\r\n";
   RXSTRING instore[2];
   RXSYSEXIT exits[3];

   /*
    * Before you can use a Rexx exit hook you have to register the hook.
    * We register an exit handler in the current executable in opposite to
    * a handler in an external library.
    *
    * ONLY USE RexxRegisterExitExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    */

   retval = RexxRegisterExitExe( "myExitHandler",
                                 (PFN) interceptor,
                                 NULL );
   if ( retval != RXEXIT_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXEXIT_??? values
       * for possible errors.
       */
      printf( "Registering of the exit hook ended with code %u\n",
              (unsigned) retval );
   }

   /*
    * Setup the list of Exit hooks. The last one must contain the value
    * RXENDLST as the sysexit_code. sysexit_name's value is irrelevant in this
    * case.
    * Every "useful" entry should have both a registered exit handler's name
    * as sysexit_name and one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC,
    * RXINI, RXTER, RXDBG, RXENV as sysexit_code.
    */
   exits[0].sysexit_name = "myExitHandler";
   exits[0].sysexit_code = RXINI;
   exits[1].sysexit_name = "myExitHandler";
   exits[1].sysexit_code = RXTER;
   exits[2].sysexit_code = RXENDLST;

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       exits,        /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   /*
    * We have to deregister the exit hook. This isn't really needed before
    * the program's end but provided as an example for proper programming.
    */
   RexxDeregisterExit( "myExitHandler", NULL );

   return (int) ( retval ? retval : rc );
}
