/*
 * This example lets a user's program intercept the input of a stream when
 * running a Rexx script within the program's environment.
 *
 * Used functions: RexxStart, RexxAllocateMemory, RexxFreeMemory,
 *                 RexxRegisterExitExe, RexxDeregisterExit
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING, RXSYSEXIT, PEXIT, RXSIOTRD_PARM
 * You should have read: step3-1.c, step2-4.c
 *
 * ALLOCATING OR FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN EXIT HANDLER SET IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Exit" stuff.
 */
#define INCL_RXSYSEXIT
#include "rexx_header.h"
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * usingTTY returns 1 if the stdin is connected to a tty.
 * 0 is returned otherwise.
 */
static int usingTTY( void )
{
   static int retval = -1;
   struct stat st;

   if ( retval != -1 )
      return retval;
   if ( fstat( fileno( stdin ), &st ) != 0 )
      return retval = 1;
   return retval = S_ISCHR( st.st_mode );
}

/*
 * setReturn copies the content into retval. A fresh buffer is allocated if
 * retval's buffer is too small. A terminating '\0' is silently appended.
 * The return value is RXEXIT_HANDLED on success or RXEXIT_RAISE_ERROR on
 * error.
 *
 * The Rexx interpreter has a return buffer of RXAUTOBUFLEN allocated
 * usually. The value is 256 in most cases.
 * 1) We don't rely on this.
 * 2) We are not allowed to free the string if it is too small. It is the
 *    property of the interpreter. It will check for a "lost" string by its
 *    own.
 * 3) Using or replacing the return string is allowed.
 * 4) For good practice we append a not counted ASCII terminator.
 * Regina uses 256 bytes, provides a line terminator but doesn't expect it.
 *
 * Allocating memory should be done with RexxAllocateMemory. This function
 * can be used with Regina and some other interpreters. It is not a known
 * function in IBM's API.
 */
static ULONG setReturn( PRXSTRING retval, const char *content )
{
   char *p;
   int len = strlen( content );

   if ( RXNULLSTRING( *retval ) ||
        ( RXSTRLEN( *retval ) <= len ) )
      p = RexxAllocateMemory( len + 1 );
   else
      p = RXSTRPTR( *retval );

   if ( p == NULL )
      return RXEXIT_RAISE_ERROR;

   memcpy( p, content, len );
   MAKERXSTRING( *retval, p, len );
   return RXEXIT_HANDLED;
}

/*
 * Have a look at "RexxExitHandler" in the header file.
 * An exit handler will get some IO requests of the Rexx interpreter. It may
 * or may not handle it.
 * main_code is one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC, RXINI, RXTER,
 * RXDBG, RXENV currently.
 * sub_code are different. Look at the define'd values in the header file
 * starting with the name of the main_code, e.g. RXSIOTRD.
 * data is a generic placeholder for the true data structure depending on
 * the main_code. We use here RXSIOTRD_PARM and have to cast data to such a
 * structure.
 * NEVER FORGET "APIENTRY" as a modifier to the function. Your stack becomes
 * confused otherwise and you'll have to understand why the return of the
 * caller of this function does a return to a random stack place.
 */
static LONG APIENTRY interceptor( LONG main_code,
                                  LONG sub_code,
                                  PEXIT data )
{
   static int first = 1;
   RXSIOTRD_PARM *arg = (RXSIOTRD_PARM *) data;

   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return RXEXIT_RAISE_ERROR or RXEXIT_NOT_HANDLED in this case.
    */
   if ( main_code != RXSIO )
      return RXEXIT_RAISE_ERROR;

   /*
    * Every sub_code must be expected and RXEXIT_NOT_HANDLED should be returned
    * for every sub_code that shouldn't be handled.
    * Anything other than "SAY" output is treated as an error here. A more
    * sophisticated logic should be implemented for the different cases.
    */
   if ( sub_code == RXSIOSAY )
      return RXEXIT_NOT_HANDLED;
   if ( sub_code != RXSIOTRD )
      return RXEXIT_RAISE_ERROR;

   if ( !usingTTY() )
   {
      char buf[2048];
      int len;

      /*
       * It is the responsibility of the exit handler to remove
       * line terminators.
       */
      if ( fgets( buf, sizeof( buf ), stdin ) != NULL )
      {
         len = strlen( buf );
         if ( len > 0 )
         {
            if ( buf[len - 1] == '\n')
            {
               len--;
               if ( ( len > 0 ) && ( buf[len - 1] == '\r') )
                  len--;
            }
            else if ( buf[len - 1] == '\r')
            {
               len--;
               if ( ( len > 0 ) && ( buf[len - 1] == '\n') )
                  len--;
            }
         }
         buf[len] = '\0';
         return setReturn( &arg->rxsiotrd_retc, buf );
      }
   }

   if ( first )
   {
      first = 0;
      printf( "RXSIOTRD input will be filled by \"INSERTED LINE\"\n");
      return setReturn( &arg->rxsiotrd_retc, "INSERTED LINE" );
   }

   /*
    * We cannot instruct the interpreter to raise a NOTREADY condition.
    * Closing the stdin file will hopefully be an equivalent when returning
    * "NOT_HANDLED".
    */
   if ( usingTTY() )
      fclose( stdin );

   return RXEXIT_NOT_HANDLED;
}

int main( int argc, char **argv )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "signal on notready\r\n"
                        "do forever\r\n"
                        "   parse pull line\r\n"
                        "   say line\r\n"
                        "   end\r\n"
                        "notready:\r\n"
                        "say '----> Finished <----'\r\n";
   RXSTRING instore[2];
   RXSYSEXIT exits[2];

   (argc = argc);

   printf( "You can redirect the input of %s to see how it\n"
           "interacts with the standard input channel.\n"
           "Use a small text file.\n", argv[0] );

   /*
    * Before you can use a Rexx exit hook you have to register the hook.
    * We register an exit handler in the current executable as opposed to
    * a handler in an external library.
    *
    * ONLY USE RexxRegisterExitExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    */

   retval = RexxRegisterExitExe( "myExitHandler",
                                 (PFN) interceptor,
                                 NULL );
   if ( retval != RXEXIT_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXEXIT_??? values
       * for possible errors.
       */
      printf( "Registering of the exit hook ended with code %u\n",
              (unsigned) retval );
   }

   /*
    * Setup the list of Exit hooks. The last one must contain the value
    * RXENDLST as the sysexit_code. sysexit_name's value is irrelevant in this
    * case.
    * Every "useful" entry should have both a registered exit handler's name
    * as sysexit_name and one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC,
    * RXINI, RXTER, RXDBG, RXENV as sysexit_code.
    */
   exits[0].sysexit_name = "myExitHandler";
   exits[0].sysexit_code = RXSIO;
   exits[1].sysexit_code = RXENDLST;

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       exits,        /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   /*
    * We have to deregister the exit hook. This isn't really needed before
    * the program's end but provided as an example for proper programming.
    */
   RexxDeregisterExit( "myExitHandler", NULL );

   return (int) ( retval ? retval : rc );
}
