/*
 * This example lets a user's program intercept the output of tracing when
 * running a Rexx script within the program's environment.
 *
 * Used functions: RexxStart, RexxFreeMemory, RexxRegisterExitExe,
 *                 RexxDeregisterExit
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING, RXSYSEXIT, PEXIT, RXSIOTRC_PARM
 * You should have read: step3-1.c, step2-4.c
 *
 * FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN EXIT HANDLER SET IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Exit" stuff.
 */
#define INCL_RXSYSEXIT
#include "rexx_header.h"
#include <stdio.h>
#include <string.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

/*
 * Have a look at "RexxExitHandler" in the header file.
 * An exit handler will get some IO requests of the Rexx interpreter. It may
 * or may not handle it.
 * main_code is one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC, RXINI, RXTER,
 * RXDBG, RXENV currently.
 * sub_code are different. Look at the define'd values in the header file
 * starting with the name of the main_code, e.g. RXSIOTRC.
 * data is a generic placeholder for the true data structure depending on
 * the main_code. We use here RXSIOTRC_PARM and have to cast data to such a
 * structure.
 * NEVER FORGET "APIENTRY" as a modifier to the function. Your stack becomes
 * confused otherwise and you'll have to understand why the return of the
 * caller of this function does a return to a random stack place.
 */
static LONG APIENTRY interceptor( LONG main_code,
                                  LONG sub_code,
                                  PEXIT data )
{
   static int first = 1;
   RXSIOTRC_PARM *arg = (RXSIOTRC_PARM *) data;

   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return RXEXIT_RAISE_ERROR or RXEXIT_NOT_HANDLED in this case.
    */
   if ( main_code != RXSIO )
      return RXEXIT_RAISE_ERROR;

   /*
    * Every sub_code must be expected and RXEXIT_NOT_HANDLED should be returned
    * for every sub_code that shouldn't be handled.
    * Anything other than "SAY" output is treated as an error here. A more
    * sophisticated logic should be implemented for the different cases.
    */
   if ( sub_code == RXSIOSAY )
      return RXEXIT_NOT_HANDLED;
   if ( sub_code != RXSIOTRC )
      return RXEXIT_RAISE_ERROR;

   printf( "RXSIOTRC output: \"%.*s\"\n",
           (int) RXSTRLEN( arg->rxsio_string ),
           RXSTRPTR( arg->rxsio_string ) );

   /*
    * Even after a "handled" event one may return "not handled" sometimes.
    * It depends on the function. A look over the parameter won't worry in
    * all cases and just inspecting data, e.g. for debugging purpose, is a
    * nice feature where RXEXIT_NOT_HANDLED is appropriate.
    *
    * RXEXIT_NOT_HANDLED instructs the interpreter to continue iterating
    * through the list of exit handlers for this main_code. If neither handler
    * returns RXEXIT_HANDLED, the default action happens which is obvious
    * in case of tracing.
    */
   if ( first )
   {
      first = 0;
      printf("(returning RXEXIT_NOT_HANDLED, all other return RXEXIT_HANDLED)\n");
      return RXEXIT_NOT_HANDLED;
   }

   return RXEXIT_HANDLED;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "trace i\r\n"
                        "say '2**256 + 1 ='\r\n"
                        "numeric digits 99\r\n"
                        "say 2**256+1\r\n";
   RXSTRING instore[2];
   RXSYSEXIT exits[2];


   /*
    * Before you can use a Rexx exit hook you have to register the hook.
    * We register an exit handler in the current executable in opposite to
    * a handler in an external library.
    *
    * ONLY USE RexxRegisterExitExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    */

   retval = RexxRegisterExitExe( "myExitHandler",
                                 (PFN) interceptor,
                                 NULL );
   if ( retval != RXEXIT_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXEXIT_??? values
       * for possible errors.
       */
      printf( "Registering of the exit hook ended with code %u\n",
              (unsigned) retval );
   }

   /*
    * Setup the list of Exit hooks. The last one must contain the value
    * RXENDLST as the sysexit_code. sysexit_name's value is irrelevant in this
    * case.
    * Every "useful" entry should have both a registered exit handler's name
    * as sysexit_name and one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC,
    * RXINI, RXTER, RXDBG, RXENV as sysexit_code.
    */
   exits[0].sysexit_name = "myExitHandler";
   exits[0].sysexit_code = RXSIO;
   exits[1].sysexit_code = RXENDLST;

   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       exits,        /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * After the return, instore[1] may be filled by "something". You should
    * free the buffer. See step2-5 for the reason. The buffer is allocated
    * by the interpreter and the caller of the interpreter should free it.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   /*
    * We have to deregister the exit hook. This isn't really needed before
    * the program's end but provided as an example for proper programming.
    */
   RexxDeregisterExit( "myExitHandler", NULL );

   return (int) ( retval ? retval : rc );
}
