/*
 * This example lets a user's program intercept the output of SAY when running
 * a Rexx script within the program's environment.
 *
 * Used functions: RexxStart, RexxRegisterExitExe, RexxDeregisterExit
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING, RXSYSEXIT, PEXIT, RXSIOSAY_PARM
 * You should have read: step2-1.c
 *
 * EACH THREAD HAS ITS OWN EXIT HANDLER SET IN MULTITHREADING ENVIRONMENTS.
 */

/*
 * You have to say that you want the "Exit" stuff.
 */
#define INCL_RXSYSEXIT
#include "rexx_header.h"
#include <stdio.h>

/*
 * Have a look at "RexxExitHandler" in the header file.
 * An exit handler will get some IO requests of the Rexx interpreter. It may
 * or may not handle it.
 * main_code is one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC, RXINI, RXTER,
 * RXDBG, RXENV currently.
 * sub_code are different. Look at the define'd values in the header file
 * starting with the name of the main_code, e.g. RXSIOSAY.
 * data is a generic placeholder for the true data structure depending on
 * the main_code. We use here RXSIOSAY_PARM and have to cast data to such a
 * structure.
 * NEVER FORGET "APIENTRY" as a modifier to the function. Your stack becomes
 * confused otherwise and you'll have to understand why the return of the
 * caller of this function does a return to a random stack place.
 */
static LONG APIENTRY interceptor( LONG main_code,
                                  LONG sub_code,
                                  PEXIT data )
{
   static int first = 1;
   RXSIOSAY_PARM *arg = (RXSIOSAY_PARM *) data;

   /*
    * It is best programming practice to check the codes even if they are
    * unexpected.
    * You should return RXEXIT_RAISE_ERROR or RXEXIT_NOT_HANDLED in this case.
    */
   if ( main_code != RXSIO )
      return RXEXIT_RAISE_ERROR;

   /*
    * Every sub_code must be expected and RXEXIT_NOT_HANDLED should be returned
    * for every sub_code that shouldn't be handled.
    * If it is expected that the script is doing tracing or reading data,
    * these cases should be handled here.
    */
   if ( sub_code != RXSIOSAY )
      return RXEXIT_RAISE_ERROR;

   printf( "RXSIOSAY output: \"%.*s\"\n",
           (int) RXSTRLEN( arg->rxsio_string ),
           RXSTRPTR( arg->rxsio_string ) );

   /*
    * Even after a "handled" event one may return "not handled" sometimes.
    * It depends on the function. A look over the parameter won't worry in
    * all cases and just inspecting data, e.g. for debugging purpose, is a
    * nice feature where RXEXIT_NOT_HANDLED is appropriate.
    *
    * RXEXIT_NOT_HANDLED instructs the interpreter to continue iterating
    * through the list of exit handlers for this main_code. If neither handler
    * returns RXEXIT_HANDLED, the default action happens which is obvious
    * in case of "SAY".
    */
   if ( first )
   {
      first = 0;
      printf("(returning RXEXIT_NOT_HANDLED, all other return RXEXIT_HANDLED)\n");
      return RXEXIT_NOT_HANDLED;
   }

   return RXEXIT_HANDLED;
}

int main( void )
{
   ULONG retval;
   SHORT rc;
   RXSYSEXIT exits[2];


   /*
    * Before you can use a Rexx exit hook you have to register the hook.
    *
    * Rexx provides two different versions for registering. The appropriate one
    * in this situation and the simpler one is RexxRegisterExitExe. The other
    * one is RexxRegisterExitDll. Like all other registering functions of
    * Rexx, the "Exe" version fetches just a function pointer, while the
    * "Dll" version needs the name of a shared library (aka DLL) and the name
    * of an exported function/procedure of the shared library.
    *
    * The "Dll" version links the shared library, fetches the procedure entry
    * of the given function name and behaves like the RexxRegisterExitExe with
    * the fetched procedure entry. The "Dll" version shall be used in all cases
    * where external libraries are used. The Rexx interpreter knows what to
    * do in cases where different registrations have been done and a
    * deregistration happens.
    *
    * One scenario where these considerations take effect is this:
    * A Rexx support package is registered by a call to RxFuncAdd. The function
    * of the shared library registers an Exit handler for some purpose. It does
    * this by RexxRegisterExitExe. This will work for the first. But supposed
    * the function is dropped, e.g. by RxFuncDrop, the Rexx interpreter may
    * decide to unlink the DLL from its working set of libraries. The next
    * hook where the exit handler is involved will let the Rexx interpreter
    * jump into outer space and will hopefully be killed by the operating system
    * due to an illegal use of memory.
    *
    * ONLY USE RexxRegisterExitExe IN THE MAIN PROGRAM OR IN A SHARED
    * LIBRARY AKA DLL WHICH IS NEVER(!!) UNLOADED.
    *
    * The other parameters of RexxRegisterExit... are the name of the hook
    * for Rexx and a private area which can be used for magic things. The name
    * is rather useful, since it is possible to register loads of handlers on
    * startup and select different exit handlers while using various calls
    * to RexxStart. The name is used case-sensitive.
    *
    * The user area is rarely useful in most cases. A value of NULL is allowed.
    * If it is set, a 8 byte buffer is copied from that address and associated
    * with the exit handler. Regina doesn't inspect the buffer content.
    *
    * RexxRegisterExitDll isn't supported by Regina including Version 3.2.
    */

   retval = RexxRegisterExitExe( "myExitHandler",
                                 (PFN) interceptor,
                                 NULL );
   if ( retval != RXEXIT_OK )
   {
      /*
       * Have a look at the Rexx include file at the various RXEXIT_??? values
       * for possible errors.
       */
      printf( "Registering of the exit hook ended with code %u\n",
              (unsigned) retval );
   }

   /*
    * Setup the list of Exit hooks. The last one must contain the value
    * RXENDLST as the sysexit_code. sysexit_name's value is irrelevant in this
    * case.
    * Every "useful" entry should have both a registered exit handler's name
    * as sysexit_name and one of RXFNC, RXCMD, RXMSQ, RXSIO, RXHLT, RXTRC,
    * RXINI, RXTER, RXDBG, RXENV as sysexit_code.
    */
   exits[0].sysexit_name = "myExitHandler";
   exits[0].sysexit_code = RXSIO;
   exits[1].sysexit_code = RXENDLST;
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "step2.rexx", /* ProgramName */
                       NULL,         /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       exits,        /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */

   /*
    * We have to deregister the exit hook. This isn't really needed before
    * the program's end but provided as an example for proper programming.
    * The first parameter is the name of an exit handler while the second
    * one, ModuleName, is ignored by Regina currently. It may be NULL for
    * default or the name of a shared library (as given when registering using
    * RexxRegisterExitDll) to limit the search to just that library.
    * Note that a duplicate registration with the same name and different
    * modules may happen and is allowed. When deregistering, the Exe version
    * is searched first for the handler. Regina uses just one list for all
    * handlers.
    */
   RexxDeregisterExit( "myExitHandler", NULL );

   return (int) ( retval ? retval : rc );
}
