/*
 * This example lets the interpreter execute some lines of Rexx code without
 * the use of any file holding the code and repeats it by using precompiled
 * code.
 *
 * Used functions: RexxStart, RexxFreeMemory
 * Used pseudo-functions: MAKERXSTRING, RXSTRPTR, RXSTRLEN, RXNULLSTRING
 * Used structures: RXSTRING
 * You should have read: step2-1.c, step2-2.c, step2-4.c
 *
 * The final parameter Exits to RexxStart is discussed in step3 ff.
 *
 * FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN MEMORY POOL IN MULTITHREADING ENVIRONMENTS.
 */
#include "rexx_header.h"
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

int main( void )
{
   ULONG retval;
   SHORT rc;
   static char *macro = "interpret arg(1)\r\n";
   char arg[128];
   int len;
   RXSTRING arg_string, instore[2];


   /*
    * You can reuse an instore macro. The macro in this file is extremely
    * short, but if an application contains several thousands of lines, it
    * is much faster to use a precompiled application.
    *
    * A precompiled version is created each time when a macro is compiled.
    * The precompiled macro is passed back in the second instore buffer.
    * The buffer will contain non-readable characters and is for the internal
    * use of the interpreter. You should always use the same interpreter to
    * replay the macro and the version should be the version that
    * compiled the macro.
    * A macro compiled on a different machine family may or may not run, nobody
    * should rely on such things.
    *
    * The compiled macro can be saved to a file or something else and can
    * be read back by another or the same application to let it run.
    *
    * NOTE: YOU DON'T HAVE ACCESS TO THE RESULT OF SOURCELINE(???) IN A
    *       COMPILED VERSION OF A MACRO IN REGINA. THIS MAY CHANGE IN THE
    *       FUTURE.
    *       This behaviour may change from interpreter to interpreter.
    */
   MAKERXSTRING( instore[0], macro, strlen( macro ) );
   MAKERXSTRING( instore[1], NULL, 0 );
   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "in memory",  /* ProgramName */
                       instore,      /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       NULL,         /* Exits */
                       &rc,          /* ReturnCode */
                       NULL );       /* Result */
   if ( retval != 0 )
      return (int) retval;

   /*
    * The script's compiled version is stored in instore[1]. We reuse it now.
    * instore[0] should point to the source string, but the script may run
    * without with the the source; SOURCELINE(???) then will not work.
    */
   printf( "A simple precompiled script occupies %d byte.\n"
           "Enter one-line expressions to test it.\n"
           "Enter the empty string to terminate the test.\n"
           "\n",
           (int) RXSTRLEN( instore[1] ) );
   while ( fgets( arg, sizeof( arg ), stdin ) != NULL )
   {
      /*
       * Chop off the line terminator and garbage.
       */
      len = strlen( arg );
      while ( len > 0 )
      {
         if ( !isspace( arg[len - 1] ) )
            break;
         len--;
      }
      if ( len == 0 )
         break;
      arg[len] = '\0';

      printf( "Passing \"%s\" for interpretation to a precompiled script.\n",
              arg );
      MAKERXSTRING( arg_string, arg, strlen( arg ) );
      retval = RexxStart( 1,            /* ArgCount */
                          &arg_string,  /* ArgList */
                          "in memory",  /* ProgramName */
                          instore,      /* Instore */
                          "Foo",        /* EnvName */
                          RXCOMMAND,    /* CallType */
                          NULL,         /* Exits */
                          &rc,          /* ReturnCode */
                          NULL );       /* Result */
      printf( "\n" );
   }

   /*
    * Cleanup and free the instore macro if it exists.
    */
   if ( !RXNULLSTRING( instore[1] ) )
      RexxFreeMemory( RXSTRPTR( instore[1] ) );

   return 0;
}
