/*
 * This example lets the interpreter execute a Rexx script and check the
 * return values.
 *
 * Used functions: RexxStart, RexxFreeMemory
 * Used pseudo-functions: RXSTRPTR, RXSTRLEN, MAKERXSTRING, RXNULLSTRING
 * Used structures: RXSTRING
 * You should have read: step2-1.c
 *
 * FREEING STRINGS IS NOT PORTABLE.
 *
 * EACH THREAD HAS ITS OWN MEMORY POOL IN MULTITHREADING ENVIRONMENTS.
 */
#include "rexx_header.h"
#include <stdio.h>
#if !defined(REXXFREEMEMORY) || defined(UNKNOWN_INTERPRETER)
# include "rexxmem.h"
#endif

int main( void )
{
   ULONG retval;
   SHORT rc;
   RXSTRING result;
   char buf[20];


   /*
    * We want to fetch the return values of a script. You should know that
    * are two different values: The number representation and the string
    * representation. Calls within a script to "return" at top level or "exit"
    * are tried to be interpreted as numbers. The number representation will
    * become 0 if the return value isn't a number.
    * Examples:
    *    "return 'Foo'" let the ReturnCode become 0
    *    "return '107'" let the ReturnCode become 107
    *
    * We want to fetch the result string. A Rexx string is a structure
    * both the true string and its length. Theoretically you can even fetch
    * string terminators, but most interpreters support it very poorly. Most
    * interpreters nul terminate the result string, though it's better not to
    * rely on it.
    *
    * Well, the container for the return string is the property of the invoker.
    * This is true in all cases. The content itself MAY be allocated by the
    * the caller and it may be ignored by the callee. This is confusing, but
    * it has advantages. Some examples will explain it.
    *
    * return value       buffer size       action
    * "short val"        10                supplied buffer used
    * "longer value"     10                supplied buffer ignored, new
    *                                      allocated
    *
    * Even a NULL pointer is allowed here, as far as a buffer size of 0 is
    * used.
    *
    * ATTENTION: IF THE INTERPRETER DECIDES TO USE A NEW BUFFER, THE CALLER
    *            SHALL FREE THE BUFFER AFTER USAGE.
    *
    * If the supplied buffer fits all requirements be the interpreter,
    * everything is perfect. The interpreter uses it. But if the supplied
    * buffer is too short, the interpreter allocates a new one of a proper
    * size and overwrites the caller's one WITHOUT FREEING THE CALLER'S BUFFER.
    *
    * This sounds inconsistent, but the advantage is that the caller may use
    * any buffer, even a static one, and just need to free the returned string
    * if it has been allocated by the interpreter.
    *
    * MAKERXSTRING fills the structure's elements properly. You can use
    * RXSTRLEN and RXSTRPTR to access the fields.
    * RXNULLSTRING tests whether a RXSTRING contains a NULL pointer or not.
    */

   MAKERXSTRING( result, buf, sizeof(buf) );  /* fill the structure result */

   retval = RexxStart( 0,            /* ArgCount */
                       NULL,         /* ArgList */
                       "step2.rexx", /* ProgramName */
                       NULL,         /* Instore */
                       "Foo",        /* EnvName */
                       RXCOMMAND,    /* CallType */
                       NULL,         /* Exits */
                       &rc,          /* ReturnCode */
                       &result );    /* Result */

   printf( "The return value of RexxStart is %d\n", (int) retval );
   printf( "The return value of the script is %d\n", (int) rc );
   printf( "The result of the script is %.*s\n",
           (int) RXSTRLEN( result ), RXSTRPTR( result ) );

   if ( (char *) (RXSTRPTR( result )) != buf )
   {
      /*
       * The structure's pointer has been overwritten. The interpreter has
       * allocated a new chunk of memory.
       * Many interpreters use different allocation schemes in different
       * OSes. Some like Regina supply a special function called RexxFreeMemory
       * for it. You can safely use your code in different systems with such
       * a function. Otherwise you have to know if the needed function is
       * called VirtualFree, GlobalFree, free, ...
       */

      if ( !RXNULLSTRING( result ) ) /* fail-safe test */
         RexxFreeMemory( RXSTRPTR( result ) );
   }

   return (int) retval;
}
