/*
 * This file is a helper file which should provide the interpreter's
 * memory allocating and deallocating functions.
 *
 * Good assumptions are
 * A) Windows:
 *    GlobalAlloc / GlobalFree or VirtualAlloc / VirtualFree
 * B) OS/2:
 *    DosAllocMem / DosFreeMem
 * C) DOS:
 *    malloc / free
 * D) unix:
 *    malloc / free
 *
 * The documentations of the current interpreter should be checked of course.
 *
 * This small header file is included if REXXALLOCATEMEMORY isn't defined.
 * RexxAllocateMemory and RexxFreeMemory are the better choice in all cases
 * if they exist.
 */

#ifdef WIN32
# define WIN32_LEAN_AND_MEAN
# include <windows.h>
# define RexxAllocateMemory(x) GlobalAlloc( GMEM_FIXED, x )
# define RexxFreeMemory(x)     GlobalFree( x )

#elif defined(OS2)
# define WIN32_LEAN_AND_MEAN
# include <os2.h>
  static void *__mem;
# define RexxAllocateMemory(x) ( DosAllocMem( &__mem, x, fPERM|PAG_COMMIT ) ? __mem : NULL )
# define RexxFreeMemory(x)     DosFreeMem( x )

#else
# include <stdlib.h>
# define RexxAllocateMemory malloc
# define RexxFreeMemory     free

#endif
