/*
 * This script detects the current Rexx interpreter in the first step.
 * Then it creates all possible binaries using the normal C compiler for
 * the current system.
 *
 * The user can choose a specific interpreter if more than one is installed.
 * A call with an invalid argument (`?' for example) shows supported values.
 *
 * "clean" as the argument performs a cleaning-up step.
 */
trace o
parse arg int

/* Supported systems */
interpreters   = 'regina rexxtrans orexx rexximc clean'
/* Libraries and compiler definitions for the various systems */
_lib.regina    = 'regina'
_def.regina    = '-DUSE_REGINA'
_lib.rexxtrans = 'rexxtrans'
_def.rexxtrans = '-DUSE_REXXTRANS'
_lib.orexx     = 'rexx'
_def.orexx     = '-DUSE_OREXX'
_lib.rexximc   = 'rexx'
_def.rexximc   = '-DUSE_REXXIMC'
/* These programs exist as examples: */
execs          = 'step1',
                 'step2-1 step2-2 step2-3 step2-4 step2-5',
                 'step3-1 step3-2 step3-3 step3-4 step3-5 step3-6 step3-7 step3-8 step3-9',
                 'step4',
                 'step5',
                 'step6',
                 'step7',
                 'step8-1 step8-2 step8-3 step8-4',
                 'step9'

/*
 * Check for a valid user supplied interpreter.
 */
if int \= '' & wordpos( int, interpreters ) = 0 then
   call abort 'Invalid interpreter. Must be one of:' interpreters


parse version ver
parse source os .
say ver os
/*
 * Determine which interpreter we are running; compile for that interpreter...
 * Only check this if we haven't explicitly specified an interpreter
 */
if int = '' then
   select
      when left( ver, 11 ) = 'REXX-Regina' then
         int = 'REGINA'
      otherwise
         call abort 'This interpreter:' ver 'not recognised'
   end
else
   int = translate( int )

select
   when os = 'WIN32' then
      do
      cc = 'cl -nologo -O'
      flags = _def.int '/out:'
      libs = _lib.int'.lib'
      exe = '.exe'
      rm = 'del'
      end
   otherwise
      do
      cc = 'gcc -O -Wall'
      flags = _def.int '-o'
      exe = ''
      libs = '-l'_lib.int
      rm = 'rm'
      end
end

/*
 * If "clean" then delete the binaries.
 * The default address target may change between the various interpreters. We
 * have to assume that the default address environment will execute the
 * commands like C's "system()" call.
 */
if int = 'CLEAN' then
   do
      do i = 1 to words( execs )
         rm word( execs, i ) || exe
      end
      return 0
   end

do i = 1 To Words( EXECS )
   prog = Word( EXECS, i )
   parse value stream( prog'.c', 'C', 'QUERY TIMESTAMP' ) with c_date c_time
   parse value stream( prog||exe, 'C', 'QUERY TIMESTAMP' ) with exe_date exe_time
   c_test = changestr( '-', c_date, '' ) c_time
   exe_test = changestr( '-', exe_date, '' ) exe_time
   if c_test > exe_test then
      do
         call charout , prog' '
         cc flags prog||exe prog'.c' libs
         if rc \= 0 then
            call abort 'error:' rc 'compiling' prog
      end
end

say 'All done!'
return 0

abort:
   parse arg msg
   say msg
   exit 1
   /* return not reached */
